function [x,xd]=thresh3_LPyr(theta,nframes,x)
%
% [x,xd] = thresh3_LPyr(theta,nframes,x)
%
% Thresholding 3D image data after taking the 3D Lapalcian
% Pyramid local FFT
%
% theta        --> threshold value ('soft' for adaptive)
% nframes      --> range of frames to display for comparison
% x (optional) --> image data (default: MissA sequence)
%

%*******************
if (nargin < 2)
	help thresh3_LPyr
	error('Bad input arguments ..');
end

%*******************
nlev    = 3;
wSize   = [8 8 8];
%theta = sqrt(2*log(n)) * std(noise);

ptime   = 0.1;

%*******************

if (exist('x'))
	disp('Using the input 3-D data variable ''x'' ..');
else
	disp('Loading the 3D Miss America sequence ..');

	load missa_complete
	%x = y(10:265,80:335,23:150);
	x = y(80:207,155:282,23:150);
	x = double(x);
	clear y
end

if (theta == 'soft')
	sigma = input('Enter the value for sigma: ');
end

%*******************

disp(sprintf('Computing %d-level 3D Laplacian pyramid ..',nlev));

pyr = computeLpyr3(x,3);

%*******************

ncEncoded = 0;

for k=1:nlev
disp(sprintf('Band number: %d',k));
	bandK      = pyr{k};

str=sprintf('  Computing local FFT with [%d %d %d] window ..', ...
wSize(1),wSize(2),wSize(3));
disp(str);
	bandK_LFFT = computeLocalFFT3(bandK,wSize,0);
disp('  Thresholding the local FFT ..');
	if (theta == 'soft')
		[bandK_LFFT,ncAbove] = softThresh3(bandK_LFFT,sigma);
	else
		[bandK_LFFT,ncAbove] = hardThresh3(bandK_LFFT,theta);
	end

	ncEncoded = ncEncoded + ncAbove;

str=sprintf('  Computing inv. local FFT with [%d %d %d] window ..',...
wSize(1),wSize(2),wSize(3));
disp(str);
	bandK      = computeInvLocalFFT3(bandK_LFFT,wSize);

	pyr{k}     = real(bandK);
end

%*******************
disp(sprintf('Computing %d-level inverse 3D Laplacian pyramid ..',nlev));

xd = reconLpyr3(pyr);

%*******************
psnrValue = computePSNR(x,xd);
disp(sprintf('PSNR of the thresholded image data is: %-6.2fdB ..',psnrValue));
disp(sprintf('Total number of encoded coefficients is: %d (%6.2f%% of %dx%dx%d)..',ncEncoded,ncEncoded/prod(size(x))*100,size(x,1),size(x,2),size(x,3)));

%*******************
disp('Displaying one of the frames ..');

for k=nframes
	subplot(1,2,1);
	showImage3(x,k,1,0);
	title(sprintf('Original frame# %d',k));

	subplot(1,2,2);
	showImage3(xd,k,1,0);
	title(sprintf('Reconstructed frame# %d',k));

	pause(ptime);
end

gtext(sprintf('PSNR: --> %5.2fdB',psnrValue));


%----------------------------------------------------------------

function [yc,ncAbove]=hardThresh3(xc,theta)

%
% yc=hardThresh3(xc,theta)
%
% Thresholds the 3D complex input 'xc' using threshold 'theta'
% (by equating to zero all inputs with magnitude below threshold)
% and returns the result in 'yc'
%
yc   = log(1+abs(xc));
mini = min(min(min(yc)));
maxi = max(max(max(yc)));

ind  = find(yc<theta);
pct  = length(ind)/length(xc(:)) * 100;

disp(sprintf('  [min=%f, max=%f]', mini,maxi));
disp(sprintf('  [theta=%f] (%6.2f%% coeffs from %dx%dx%d below thr)',...
	theta, pct, size(xc,1),size(xc,2),size(xc,3)));

xc(ind) = 0;
yc = xc;

ncAbove = (1-pct/100) * prod(size(xc));


%----------------------------------------------------------------

function [yc,ncAbove]=softThresh3(xc,sigma)

%
% yc=softThresh3(xc,sigma)
%
% Soft thresholds the 3D complex input 'xc' using noise s.d.
% (by equating to zero all inputs with magnitude below threshold)
% and returns the result in 'yc'
%
% theta = sqrt(2*log(n)) * sigma
%
yc    = log(1+abs(xc));
mini  = min(min(min(yc)));
maxi  = max(max(max(yc)));

%sigma
theta = sqrt(2*log(prod(size(xc))));
theta = theta * ( 0.46*log10(sigma) + 0.92 );
%theta = theta * 1.25
%theta = 0.75*maxi;

ind   = find(yc<theta);
pct   = length(ind)/length(xc(:)) * 100;

disp(sprintf('  [min=%f, max=%f]', mini,maxi));
disp(sprintf('  [theta=%f] (%6.2f%% coeffs from %dx%dx%d below thr)',...
	theta, pct, size(xc,1),size(xc,2),size(xc,3)));

xc(ind) = 0;
yc = xc;

ncAbove = (1-pct/100) * prod(size(xc));




