% [PYR, INDICES] = buildLpyr3(x, HEIGHT, FILT1, FILT2, EDGES)
%
% Construct a Laplacian pyramid on 3-d volume 'x'.
%
% HEIGHT (optional) specifies the number of pyramid levels to build. Default
% is 1+maxPyrHt(size(x),size(FILT)).  You can also specify 'auto' to
% use this value.
%
% FILT1 (optional) can be a string naming a standard filter (see
% namedFilter), or a vector which will be used for (separable)
% convolution.  Default = 'binom5'.  FILT2 specifies the "expansion"
% filter (default = filt1).  EDGES specifies edge-handling, and
% defaults to 'reflect1' (see corrDn).
%
% PYR is a vector containing the N pyramid subbands, ordered from fine
% to coarse.  INDICES is an Nx2 matrix containing the sizes of
% each subband.  This is compatible with the MatLab Wavelet toolbox.

% originally written by Eero Simoncelli, 6/96.
% modified first for 3-d by Nasir Rajpoot, 11/02.

function [pyr,pind] = buildLpyr3(x, height, filt1, filt2, edges)

if (nargin < 2)
  disp('At least first two arguments should be provided ..');
  return
end

size_x = size(x);

%------------------------------------------------------------
%% OPTIONAL ARGS:
%------------------------------------------------------------

if (exist('filt1') ~= 1)
  filt1 = 'binom5';
end
 
if isstr(filt1)
  filt1 = namedFilter(filt1);
end

if ( (size(filt1,1) > 1) & (size(filt1,2) > 1) )
  error('FILT1 should be a 1D filter (i.e., a vector)');
else
  filt1 = filt1(:);
end

if (exist('filt2') ~= 1)
  filt2 = filt1;
end

if isstr(filt2)
  filt2 = namedFilter(filt2);
end

if ( (size(filt2,1) > 1) & (size(filt2,2) > 1) )
  error('FILT2 should be a 1D filter (i.e., a vector)');
else
  filt2 = filt2(:);
end

max_ht = 1 + maxPyrHt3(size_x, max(size(filt1,1), size(filt2,1)));
if ( (exist('height') ~= 1) | (height == 'auto') )
  height = max_ht;
else
  if (height > max_ht)
    error(sprintf('Cannot build pyramid higher than %d levels.',max_ht));
  end
end

if (exist('edges') ~= 1)
  edges= 'reflect1';
end

%------------------------------------------------------------
len_f1  = length(filt1);
f1_dim1 = reshape(filt1, [len_f1 1 1]);
f1_dim2 = reshape(filt1, [1 len_f1 1]);
f1_dim3 = reshape(filt1, [1 1 len_f1]);
%------------------------------------------------------------
len_f2  = length(filt2);
f2_dim1 = reshape(filt2, [len_f2 1 1]);
f2_dim2 = reshape(filt2, [1 len_f2 1]);
f2_dim3 = reshape(filt2, [1 1 len_f2]);
%------------------------------------------------------------

if (height <= 1)

  pyr = x(:);
  pind = size_x;

else

  lo = corrDn3(x , f1_dim3, edges, [1 1 2], [1 1 1]);
  int_sz1 = size(lo);
  lo = corrDn3(lo, f1_dim2, edges, [1 2 1], [1 1 1]);
  int_sz2 = size(lo);
  lo = corrDn3(lo, f1_dim1, edges, [2 1 1], [1 1 1]);

  [npyr,nind] = buildLpyr3(lo, height-1, filt1, filt2, edges);

  hi = convUp3(lo, f2_dim1, edges, [2 1 1], [1 1 1], int_sz2);
  hi = convUp3(hi, f2_dim2, edges, [1 2 1], [1 1 1], int_sz1);
  hi = convUp3(hi, f2_dim3, edges, [1 1 2], [1 1 1], size_x);

  hi = x - hi;

  pyr = [hi(:); npyr];
  pind = [size_x; nind];

end
 
