function image = readGipl(filename)
%   readGipl:  Read image slices from file
%
%   image = readGipl('filename')
%
%   filename - input image name (eg '/bigdaily/foo.gipl')
%
%   The function reads a gipl image from file and return an image brick
%
%   Author: Carl-Fredrik Westin, SPL, Harvard   Date: 970518
%

if nargin < 1
 help readGipl;
 return;
end;

[fid,message] = fopen(filename,'r');
if (fid==-1)
   disp(message);
   return
end

% ***************************************************
% read the giple header
% *************************************************** 

% image size, and number of phases (volumes)
im.dim = fread(fid,4,'short','b');

% patent description
im.imageType =  fread(fid,1,'short','b');

% voxel dimensions
im.voxDim   =  fread(fid,4,'float','b');

% patent description
im.patDesc =  fread(fid,80,'char','b');

% misc
im.matrixElements  = fread(fid,12,'float','b');
im.identifier      = fread(fid,1,'int','b');
im.spare           = fread(fid,28,'char','b');
im.orientationFlag = fread(fid,1,'char','b');
im.flag1           = fread(fid,1,'char','b');

% min and max voxel values
im.min             = fread(fid,1,'double','b');
im.max             = fread(fid,1,'double','b');

% misc
im.origin          = fread(fid,4,'double','b');
im.pixvalOffset    = fread(fid,1,'float','b');
im.pixvalCal       = fread(fid,1,'float','b');
im.userDef1        = fread(fid,1,'float','b');
im.userDef2        = fread(fid,1,'float','b');

% magic number, should be 719555000 for a gipl image
im.magicNumber     = fread(fid,1,'int','b');

if(~(im.magicNumber==719555000))
 error('Sorry, this is not a giple file');
 return
end;

% ***************************************************
% rearrange the header information into a new header
% that makes more sense and read the data
% ***************************************************

image.hgh  = im.dim(2);
image.wid  = im.dim(1);
image.dep  = im.dim(3);

if(im.dim(4)<2)
 image.imDim  = [im.dim(2) im.dim(1) im.dim(3)]; 
else
 image.imDim  = [im.dim(2) im.dim(1) im.dim(3) im.dim(4)]; 
end

image.imSize = prod(im.dim(1:3)); %derived

if(im.voxDim(4)<2)
  image.voxDim  = [im.voxDim(1) im.voxDim(2) im.voxDim(3)]; 
else
  image.voxDim  = [im.voxDim(1) im.voxDim(2) ...
                   im.voxDim(3) im.voxDim(4)]; 
end

image.min       = im.min;          
image.max       = im.max;            

image.data = fread(fid, prod(im.dim(1:3)) ,'short','b');
image.data = reshape(image.data,[image.wid image.hgh image.dep]);
image.data = permute(image.data,[2 1 3]);

% put less used information one level down
image.misc.imageType       = im.imageType;
image.misc.patientDesc     = char(im.patDesc)';
image.misc.matrixElements  = im.matrixElements;
image.misc.identifier      = im.identifier;
image.misc.spare           = char(im.spare);
image.misc.scanOrientation = im.orientationFlag;
image.misc.flag1           = im.flag1;
image.misc.origin          = [im.origin(1) im.origin(2) ...
                              im.origin(3) im.origin(4)];
image.misc.pixvalOffset    = im.pixvalOffset;
image.misc.pixvalCal       = im.pixvalCal;
image.misc.userDef1        = im.userDef1;
image.misc.userDef2        = im.userDef2;
image.misc.magicNumber     = im.magicNumber;





