function [basis,value] = Best2dBasis(sqtree,D)
% Best3dBasis -- Coifman-Wickerhauser Best-3d-Basis Algorithm
%  Usage
%    [btree,vtree] = Best2dBasis(sqtree,D)
%  Inputs
%    sqtree   stat-quadtree (output by Calc3dStatTree)
%    D        maximum depth of tree-search
%  Outputs
%    btree    basis-quadtree of best basis
%    vtree    value of components of best basis
%             vtree(1) holds value of best basis
%
%  Description
%    The best-basis algorithm is used to pick out the ``best'' 
%    basis from all the possible bases in the packet table.
%    Here ``best'' means minimizing an additive measure of
%    information, called entropy by Coifman and Wickerhauser.
%
%    Once the stat-quadtree of entropy values is created, Best3dBasis
%    selects the best basis using the pruning algorithm described in
%    Wickerhauser's book.

%  Examples
%    % get best basis for ``image''
%    % use Coiflet 3 filter and CW Entropy
%    [n,D] = quadlength(volume);
%    qmf = MakeONFilter('Coiflet',3);
%    sqtree = Calc3dStatTree('WP',volume,D,qmf,'Entropy');
%    [btree,vtree] = Best3dBasis(sqtree,D);
%
%  Algorithm
%    Yale University has filed a patent application for this algorithm.
%    Commercial Development based on this algorithm should be cleared
%    by Yale University. Contact them for licensing information.
%
%  See Also
%    Calc3dStatTree, FPT3_CP, FPT3_WP
%
%  References
%    Wickerhauser, M.V.  _Adapted_Wavelet_Analysis_.  AK Peters (1994).
%

        global WLVERBOSE
	basis = zeros(size(sqtree));
	value = sqtree;
	for d=D-1:-1:0,
	   for bx=0:(2^d-1),
	   	for by=0:(2^d-1),   % scan tree, bottom->top left->right scan
         for bz=0:(2^d-1),
	   		vparent = sqtree(onode(d,bx,by,bz));     % mod
			vchild  = value(onode(d+1,2*bx  ,2*by  , 2*bz)) + ...
			          value(onode(d+1,2*bx+1,2*by  , 2*bz)) + ...
					  value(onode(d+1,2*bx  ,2*by+1, 2*bz)) + ...
					  value(onode(d+1,2*bx+1,2*by+1, 2*bz)) + ...
                      value(onode(d+1,2*bx  ,2*by  , 2*bz+1)) + ...
			          value(onode(d+1,2*bx+1,2*by  , 2*bz+1)) + ...
					  value(onode(d+1,2*bx  ,2*by+1, 2*bz+1)) + ...
					  value(onode(d+1,2*bx+1,2*by+1, 2*bz+1));

			if(vparent <= vchild),  % if parent better than children, RU486
				  basis(onode(d,bx,by,bz)) = 0;
				  value(onode(d,bx,by,bz)) = vparent;
			else
				  basis(onode(d,bx,by,bz)) = 1;
				  value(onode(d,bx,by,bz)) = vchild;
			end
         end
		end
	   end
	 end
	 if strcmp(WLVERBOSE,'Yes') 
	   fprintf('best basis %g \n',value(1))
	 end
	 
%
% Copyright (c) 1993. Jonathan Buckheit and David Donoho
% 
    
% Modified 2003, Zhen Yao
    
%   
% Part of WaveLab Version 802
% Built Sunday, October 3, 1999 8:52:27 AM
% This is Copyrighted Material
% For Copying permissions see COPYING.m
% Comments? e-mail wavelab@stat.stanford.edu
%   
    
