function Ent = CalcEntropy(object,Entropy,par,D)
% CalcEntropy -- Calculate entropy number of array
%  Usage
%    Ent = CalcEntropy(object,ent[,par])
%  Inputs
%    object   1-d or 2-d object
%    ent      type of entropy to use: options are
%                'Entropy' -- Coifman-Wickerhauser
%                'Log'     -- sum log(|th_i|)
%                'l^p'     -- sum |th_i|^p, 0 < p < 2, p = par
%                'N(eps)'  -- #>= eps, eps = par
%                'Risk'    -- sum min(th_i^2,eps^2), eps=par
%                'Sum'     -- sum th_i
%                'SURE'    -- SURE(Thresholding), thresh = par
%                'Bayes'   -- Best basis on Bayesian, par = [sigma]
%    par      extra parameter, depends on type of entropy
%  Outputs
%    Ent      Entropy of object
%
%  Description
%    It is traditional to normalize the object to norm 1 before
%    calling this routine.  This routine does NO pre-scaling.
%
%  See Also
%    Calc2dStatTree
%    

	if strcmp(Entropy,'Entropy'),
		p = (object(:) ).^2;
		Ent = - sum(sum(p .* log(eps+p)));
    elseif strcmp(Entropy, 'Bayes'),
        subband_std = std(object(:));

        m = length(object(:));
        sigma_subband = sqrt(max(subband_std ^ 2 - par^2, 0));
        beta = sqrt(log(m/3));
        if (sigma_subband ~= 0)
            thresh = (par^2) / (sigma_subband) * beta;
        else 
         %  disp(m);
            thresh = max(object(:))+1;
        end
        object = HardThresh(object,  thresh);
		p = (object(:) ).^2;
%        Ent = -std(p);
	%	Ent =  -sum(sum(p .* log(eps+p)));
        Ent = sum(log(eps+p));
%          Ent = 0;
%          for i=1:m,
%              if (abs(object(i))^2 <= thresh^2)
%                  Ent = Ent+ abs(object(i))^2  - par^2;
%              else Ent = Ent+par^2;
%              end
%          end
%          Ent=-Ent;
	elseif strcmp(Entropy,'Log')
		p = abs(object(:) );
		Ent = sum(log(eps+p));
	elseif strcmp(Entropy,'l^p'),
		% par = p = exponent
		p = abs(object(:) );
		Ent = sum(p .^par);
	elseif strcmp(Entropy,'N(eps)'),
		% par = eps
		p = abs(object(:));
		Ent = sum( p > par );
	elseif strcmp(Entropy,'Risk'),
		p = object(:).^2;
                m = prod(size(object));
                m = m * log2(m);
                lamb = par * (1 + sqrt(2 * log(m)));
		Ent = sum(min(p,lamb^2));
	elseif strcmp(Entropy,'Sum'),
		p = object(:);
		Ent = sum(p);
	elseif strcmp(Entropy,'SURE'),
		tt = par^2;  % par = threshold
		dim = length(object(:));
		p = object(:).^2;
		ngt = sum( p > tt);
		nlt = dim - ngt;
		slt = sum( p .* ( p <= tt )); 
		Ent = dim - 2*nlt + tt*ngt + slt;
	else
		disp('in CalcEntropy: unknown Entropy request')
		fprintf('Request=<<%s>>\n',Entropy)
		disp('I only know how to do:')
		disp(' Entropy, Log, l^p, N(eps), Risk, SURE ')
	end

%
% Copyright (c) 1993-5, Jonathan Buckheit and David Donoho
%
    
    
%   
% Part of WaveLab Version 802
% Built Sunday, October 3, 1999 8:52:27 AM
% This is Copyrighted Material
% For Copying permissions see COPYING.m
% Comments? e-mail wavelab@stat.stanford.edu
%   
    
