package lrc;

import java.util.*;
import lrc.Turn.LeftRight;

/**
 * Contains static methods for the computation of LR-coefficients and the enumeration of hive flows.
 * @author ciken
 */
public final class LRCCompute {
private LRCCompute() {};

	/**
	 * checks, if two turns lie in the same hive triangle.
	 */
	private static boolean lieInSameTriangle(Turn t1, Turn t2) {
		HashSet<ThroughputPosition> htp1 = new HashSet<ThroughputPosition>();
		htp1.add(t1.getStartPos());
		htp1.add(t1.getStartPos().goccw());
		htp1.add(t1.getStartPos().gocw());
		htp1.add(t1.getStartPos().reverse());
		htp1.add(t1.getStartPos().goccw().reverse());
		htp1.add(t1.getStartPos().gocw().reverse());
		HashSet<ThroughputPosition> htp2 = new HashSet<ThroughputPosition>();
		htp2.add(t2.getStartPos());
		htp2.add(t2.getStartPos().goccw());
		htp2.add(t2.getStartPos().gocw());
		htp2.add(t2.getStartPos().reverse());
		htp2.add(t2.getStartPos().goccw().reverse());
		htp2.add(t2.getStartPos().gocw().reverse());
		return htp1.equals(htp2);
	}

	/**
	 * checks, if a turn "t1" lies in a set of turns "it".
	 */
	private static boolean lieInSameTriangle(Turn t1, Iterable<Turn> it) {
		for (Turn t : it) {
			if (lieInSameTriangle(t1, t)) return true;
		}
		return false;
	}

	/**
	 * Finds a shortest path from the end of securepath to the start of securepath without sharing hive triangles with securepath.
	 */
	private static Vector<Turn> findShortestPath(Flow f, Vector<Turn> securepath) {
		Turn startTurn = securepath.lastElement();
		Turn endTurn = securepath.firstElement();
		LinkedList<Turn> toExpand = new LinkedList<Turn>();
		toExpand.add(startTurn);
		HashSet<Turn> expanded = new HashSet<Turn>();
		HashMap<Turn,Turn> predecessor = new HashMap<Turn,Turn>();
		Vector<Turn> path = null;
		while (!toExpand.isEmpty()) {
			Turn t = toExpand.remove();
			expanded.add(t);
			for (Turn.LeftRight course : Turn.LeftRight.values()) {
				Turn t1 = t.nextTurn(course);
				if (!canContinueTurnInResidual(t, course, f)) continue;
				
				//<eliminate subtleties at the start and end of securepath>
				if (t1.equals(new Turn(securepath.firstElement().getStartPos().reverse().goccw().reverse(),LeftRight.LEFT))) continue;
				if (t1.equals(new Turn(securepath.firstElement().getStartPos().reverse().gocw().reverse(),LeftRight.RIGHT))) continue;
				if (t1.equals(new Turn(securepath.lastElement().getEndPos().goccw().reverse(),LeftRight.LEFT))) continue;
				if (t1.equals(new Turn(securepath.firstElement().getStartPos().gocw().reverse(),LeftRight.RIGHT))) continue;
				//</eliminate subtleties at the start and end of securepath>
				
				//check if it is valid to continue turn t with turn t1
				if (t1.equals(endTurn)) {
					//found a path
					path = new Vector<Turn>();
					t1=t;
					path.add(t);
					while (predecessor.containsKey(t1)) {
						t1 = predecessor.get(t1);
						path.add(t1);
					}
					return path;
				}
				if (!expanded.contains(t1) && !toExpand.contains(t1)) {
					if (!lieInSameTriangle(t1, securepath)) {
						if (t1.course==LeftRight.LEFT) {
							try {
								if (f.getSlack(t1.getStartPos().gocw())==1) {
									if (securepath.contains(new Turn(t1.getStartPos().gocw().goccw().reverse(),LeftRight.LEFT))) continue;
								}
							} catch (OutOfTriangleException e) {}
						}
						predecessor.put(t1, t);
						toExpand.add(t1);
					}
				}
			}
		}
		return path;
	}

	/**
	 * Checks whether a turn can be continued in a specific direction, according to the rules of the residual network.
	 */
	private static boolean canContinueTurnInResidual(Turn t, LeftRight lr, Flow f) {
		Turn t1 = t.nextTurn(lr);
		if (t1.liesInBigTriangle(f.n)) {
			if (lr==Turn.LeftRight.LEFT) {
				try {
					if (f.getSlack(t.nextTurn(Turn.LeftRight.RIGHT).getEndPos())==0) return false;
				} catch (OutOfTriangleException e) {}
			} else if (t.course==Turn.LeftRight.RIGHT && lr==Turn.LeftRight.RIGHT) {
				try {
					if (f.getSlack(t.getEndPos())==0) return false;
				} catch (OutOfTriangleException e) {}
			}
			return true;
		} else return false;
	}
	
	/**
	 * recursively compute the neighbors (in the broader sense) of a flow g when a part of a secure cycle is given.
	 */
	private static HashSet<Flow> computeNeighbors(Flow g, Vector<Turn> securepath) {
		HashSet<Flow> ret = new HashSet<Flow>();
		for (LeftRight lr : LeftRight.values()) {
			Turn t = securepath.lastElement().nextTurn(lr);
			if (canContinueTurnInResidual(securepath.lastElement(), lr, g) && !lieInSameTriangle(t, securepath)) {
				Vector<Turn> newpath = new Vector<Turn>();
				newpath.addAll(securepath);
				newpath.add(newpath.lastElement().nextTurn(lr));
				Vector<Turn> path = findShortestPath(g, newpath);
				if (path!=null) ret.addAll(computeNeighbors(g,newpath));
			}
		}
		if (ret.isEmpty()) {
			if (securepath.size()<3) return ret;
			//construct one neighbor here:
			Flow f = new Flow(g);
			Vector<Turn> newpath = new Vector<Turn>();
			{
				Vector<Turn> path = findShortestPath(f, securepath);
				//securepath + path give the result!
				Collections.reverse(path);
				if (path.size()>0) path.remove(0);
				newpath.addAll(securepath);
				newpath.addAll(path);
			}
			for (Turn t : newpath) {
				f.set(t.getStartPos(), f.get(t.getStartPos())+1);
			}
			f.fix();
			ret.add(f);
		}
		return ret;
	}
	
	/**
	 * All neighbors in the broader sense are computed.
	 * Not only those who are neighbors due to a secure cycle on the honeycomb graph G.
	 */
	public static HashSet<Flow> computeNeighbors(Flow g) {
		HashSet<Flow> ret = new HashSet<Flow>();
		for (int startrow=0;startrow<=g.n;startrow++) {
			for (int startcol=0;startcol<=startrow;startcol++) {
				for (HexagonalDirection startdir : HexagonalDirection.values()) {
					for (LeftRight startlr : LeftRight.values()) {
						ThroughputPosition startpos = new ThroughputPosition(new RowAndColumn(startrow, startcol), startdir);
						Turn startturn = new Turn(startpos,startlr);
						Vector<Turn> securepath = new Vector<Turn>();
						securepath.add(startturn);
						Vector<Turn> path = findShortestPath(g, securepath);
						if (path!=null) {
							ret.addAll(computeNeighbors(g, securepath));
						}
					}
				}
			}
		}
		return ret;
	}
	

	
	
	/**
	 * For the computation with threshold:
	 */
	public static HashSet<Flow> flowCache = null;
	/**
	 * recursively compute the neighbors (in the broader sense) of a flow g when a part of a secure cycle is given.
	 * Return when the neighbor threshold is reached.
	 */
	private static void computeNeighbors(Flow g, Vector<Turn> securepath, long threshold) {
		if (flowCache.size()>threshold) return;
		for (LeftRight lr : LeftRight.values()) {
			Turn t = securepath.lastElement().nextTurn(lr);
			if (canContinueTurnInResidual(securepath.lastElement(), lr, g) && !lieInSameTriangle(t, securepath)) {
				Vector<Turn> newpath = new Vector<Turn>();
				newpath.addAll(securepath);
				newpath.add(newpath.lastElement().nextTurn(lr));
				Vector<Turn> path = findShortestPath(g, newpath);
				if (path!=null) computeNeighbors(g,newpath,threshold);
				if (flowCache.size()>threshold) return;
			}
		}
		if (securepath.size()<3) return;
		//construct one neighbor here:
		Flow f = new Flow(g);
		Vector<Turn> newpath = new Vector<Turn>();
		{
			Vector<Turn> path = findShortestPath(f, securepath);
			//securepath + path give the result!
			Collections.reverse(path);
			if (path.size()>0) path.remove(0);
			newpath.addAll(securepath);
			newpath.addAll(path);
		}
		for (Turn t : newpath) {
			f.set(t.getStartPos(), f.get(t.getStartPos())+1);
		}
		f.fix();
		flowCache.add(f);
	}
	
	/**
	 * Compute only new neighbors. A threshold is used to speed up the algorithm.
	 */
	public static HashSet<Flow> computeNeighbors(Flow g, long threshold) {
		if (flowCache.size()>threshold) return flowCache;
		for (int startrow=0;startrow<=g.n;startrow++) {
			for (int startcol=0;startcol<=startrow;startcol++) {
				for (HexagonalDirection startdir : HexagonalDirection.values()) {
					for (LeftRight startlr : LeftRight.values()) {
						ThroughputPosition startpos = new ThroughputPosition(new RowAndColumn(startrow, startcol), startdir);
						Turn startturn = new Turn(startpos,startlr);
						Vector<Turn> securepath = new Vector<Turn>();
						securepath.add(startturn);
						Vector<Turn> path = findShortestPath(g, securepath);
						if (path!=null) {
							computeNeighbors(g, securepath,threshold);
							if (flowCache.size()>threshold) return flowCache;
						}
					}
				}
			}
		}
		return flowCache;
	}
	
	
	/**
	 * Computes hive flows and terminates when there are more than max elements to return.
	 * @param f a copy of this flow will be used. The parameter will not be changed.
	 */
	public static HashSet<Flow> compute(Vector<Integer> lambda, Vector<Integer> mu, Vector<Integer> nu, Flow f, long max) {
		Flow fixf = new Flow(f);
		fixf.fix();
		HashSet<Flow> expanded = new HashSet<Flow>();
		HashSet<Flow> toExpand = new HashSet<Flow>();
		toExpand.add(fixf);
		while (!toExpand.isEmpty() && expanded.size()+toExpand.size()<=(max==-1?Long.MAX_VALUE:max)) {
			Flow g = toExpand.iterator().next();
			toExpand.remove(g);
			expanded.add(g);
			for (Flow neighbor : computeNeighbors(g)) {
				if (!expanded.contains(neighbor)) toExpand.add(neighbor);
			}
		}
		expanded.addAll(toExpand);
		return expanded;
	}
}
