#include "straighten.h"

extern inline void dictionary_straighten(struct tableau * tableau, struct shape_data_c * s_data);

extern inline void column_straighten(struct tableau * tableau, struct shape_data_c * s_data);

extern inline int get_standardization_of_colstraightened(struct tableau * tableau_arr, struct sstd_data_c * sstd_data, struct shape_data_c * s_data, uint8_t * work);

void set_conjugate(uint32_t * conjugate, uint32_t * shape, uint32_t shape_length) {
	int num, index = 0;

	for(int col = 0; col < shape[0]; col++) {
		num = 0;
		for(int row = 0; row < shape_length; row++) {
			if(shape[row] > col) {
				num += 1;
			}
		}
		if(num > 0) {
			conjugate[index] = num;
			index += 1;
		}
	}
}

void set_shape_data_components(struct shape_data_c * s_data) {
	int pos = 0, col=0, row=0;
	for(col=0; col < s_data[0].shape[0]; col++) {
		for(row=0; row < s_data[0].conjugate[col]; row++) {
			s_data[0].col_row_to_box[col * s_data[0].shape_length + row] = pos;
			s_data[0].row_col_to_box[row * s_data[0].shape[0] + col] = pos;
			s_data[0].box_to_row[pos] = row;
			s_data[0].box_to_col[pos] = col;
			pos += 1;
		}
	}
	pos = 0;
	for(row=0; row < s_data[0].conjugate[0]; row++) {
		for(col=0; col < s_data[0].shape[row]; col++) {
			s_data[0].roworder_to_box[pos] = s_data[0].col_row_to_box[col * s_data[0].shape_length + row];
            pos += 1;
		}
	}
	s_data[0].box_to_row[s_data[0].num_boxes] = 0;
    s_data[0].box_to_col[s_data[0].num_boxes] = s_data[0].box_to_col[s_data[0].num_boxes-1]+1;
}

void construct_shape_data_c(struct shape_data_c * s_data, uint32_t * shape, uint32_t shape_length) {
	s_data[0].shape = (uint32_t*) calloc(shape_length, sizeof(uint32_t));
	s_data[0].shape_length = shape_length;
	s_data[0].num_boxes = 0;
	for(int row = 0; row < shape_length; row++) {
		s_data[0].shape[row] = shape[row];
		s_data[0].num_boxes += shape[row];
	}

    s_data[0].shape_string = (char*) calloc(128, sizeof(char));
    shape_to_string(s_data[0].shape, s_data[0].shape_length, s_data[0].shape_string);

	s_data[0].num_packed_boxes = (s_data[0].num_boxes / 2) + (s_data[0].num_boxes % 2);
	s_data[0].conjugate = (uint32_t*) calloc(s_data[0].shape[0], sizeof(uint32_t));
	set_conjugate(s_data[0].conjugate, s_data[0].shape, s_data[0].shape_length);

	uint32_t rectangle_size = s_data[0].shape[0] * s_data[0].shape_length;

	s_data[0].col_row_to_box = (uint32_t*) calloc(rectangle_size, sizeof(uint32_t));
    s_data[0].row_col_to_box = (uint32_t*) calloc(rectangle_size, sizeof(uint32_t));
    s_data[0].box_to_row = (uint32_t*) calloc(s_data[0].num_boxes+1, sizeof(uint32_t));
    s_data[0].box_to_col = (uint32_t*) calloc(s_data[0].num_boxes+1, sizeof(uint32_t));
    s_data[0].roworder_to_box = (uint32_t*) calloc(s_data[0].num_boxes, sizeof(uint32_t));

    set_shape_data_components(s_data);
    
    s_data[0].col_start_single_rows = 0;
    s_data[0].box_start_single_rows = 0;

    for(int col = 0; col < s_data[0].shape[0]; col++) {
    	if(s_data[0].conjugate[col] > 1) {
    		s_data[0].col_start_single_rows = col + 1;
    		if(col < (s_data[0].shape[0]-1)) {
				s_data[0].box_start_single_rows = s_data[0].row_col_to_box[col+1];
    		}
    		else {
    			s_data[0].box_start_single_rows = s_data[0].num_boxes;
    		}
    	}
    }

    s_data[0].unique_columns = (struct columns_iterator*) calloc(1, sizeof(struct columns_iterator));
    s_data[0].unique_columns[0].length = 1;
    for(int col = 1; col < s_data[0].shape[0]; col++) {
    	if(s_data[0].conjugate[col] != s_data[0].conjugate[col-1]) {
    		s_data[0].unique_columns[0].length++;
    	}
    }
    s_data[0].unique_columns[0].unique_column_start = (uint32_t*) calloc(s_data[0].unique_columns[0].length, sizeof(uint32_t));
    s_data[0].unique_columns[0].unique_column_number = (uint32_t*) calloc(s_data[0].unique_columns[0].length, sizeof(uint32_t));
    uint32_t arr_position = 0;
    uint32_t arr_num = 0;
    s_data[0].unique_columns[0].unique_column_start[0] = 0;
    for(int col = 1; col < s_data[0].shape[0]; col++) {
    	arr_num++;
    	if(s_data[0].conjugate[col] != s_data[0].conjugate[col-1]) {
    		s_data[0].unique_columns[0].unique_column_number[arr_position] = arr_num;
    		arr_num = 0;
    		arr_position++;
    		s_data[0].unique_columns[0].unique_column_start[arr_position] = col;
    	}
    }
    arr_num++;
    s_data[0].unique_columns[0].unique_column_number[arr_position] = arr_num;
}

void destruct_shape_data_c(struct shape_data_c * s_data) {
	free(s_data[0].shape);
    s_data[0].shape = NULL;
	free(s_data[0].conjugate);
    s_data[0].conjugate = NULL;
	free(s_data[0].col_row_to_box);
    s_data[0].col_row_to_box = NULL;
	free(s_data[0].row_col_to_box);
    s_data[0].row_col_to_box = NULL;
	free(s_data[0].box_to_row);
    s_data[0].box_to_row = NULL;
	free(s_data[0].box_to_col);
    s_data[0].box_to_col = NULL;
	free(s_data[0].roworder_to_box);
    s_data[0].roworder_to_box = NULL;
    free(s_data[0].unique_columns[0].unique_column_start);
    free(s_data[0].unique_columns[0].unique_column_number);
    free(s_data[0].unique_columns);
    s_data[0].unique_columns = NULL;

    free(s_data[0].shape_string);
    s_data[0].shape_string = NULL;

    s_data[0].num_boxes = 0;
    s_data[0].num_packed_boxes = 0;
    s_data[0].shape_length = 0;
    s_data[0].shape_index = 0;
    s_data[0].box_start_single_rows = 0;
    s_data[0].col_start_single_rows = 0;
}


int set_sstd_threaded_helper(struct tableau ** tableau_array, int32_t * max_num_sstd, uint8_t * current_content, struct tableau * current_tableau, struct sstd_data_c * sstd_data, struct shape_data_c * s_data, int box, int max_pos, int max_box) {
	if (box == max_box) {
        for(int b = 0; b < max_box; b++) {
            tableau_array[0][max_pos].entries[b] = current_tableau[0].entries[b];
        }
        max_pos++;
        if(max_pos == *max_num_sstd) {
            //need to enlarge the storage tableau array
            int32_t old_size = *max_num_sstd;
            *max_num_sstd = (*max_num_sstd)*2;
            realloc_tableau_bulk_contiguous(tableau_array, old_size, *max_num_sstd, s_data);            
        }
        return max_pos;
    }

    int col = s_data[0].box_to_col[box];
    int row = s_data[0].box_to_row[box];

    int num_boxes_above_right = 0;
    for(int c = col+1; c < s_data[0].shape[0]; c++) {
    	num_boxes_above_right += straighten_min_macro(row, s_data[0].conjugate[c]);
    }

    int current_content_sum = 0;
    for(int c_val = 0; c_val < sstd_data[0].max_filling_value; c_val++) {
    	// this checks that adding the current content value would not exceed the maximum number of this value allowed
    	if(current_content[c_val] != 0) {
    		// this checks that the number of boxes that are strictly above and to the right can fit the content values less than the current box
    		// this eliminates the vast number of dead end branches
    		if(num_boxes_above_right >= current_content_sum) {
    			// this checks that the current content value is not to great to allow the column to be filled
	        	if (((c_val + 1) + (s_data[0].conjugate[col] - row - 1)) <= sstd_data[0].max_filling_value) {
	        		// this checks that the values will be weakly increasing in rows
	            	if(col == 0  || current_tableau[0].entries[s_data[0].col_row_to_box[((col-1)*s_data[0].shape_length) + row]] <= c_val + 1) {
	            		// this checks that values will be strictly increasing in columns
	                	if (row==0 || current_tableau[0].entries[box-1] < c_val + 1) {
	                		// the current content value passes all the checks, recurse to the next box
	                    	current_content[c_val] -= 1;
	                        current_tableau[0].entries[box] = c_val + 1;
	                        max_pos = set_sstd_threaded_helper(tableau_array, max_num_sstd, current_content, current_tableau, sstd_data, s_data, box+1, max_pos, max_box);
	                        current_content[c_val] += 1;
	                    }
	                }
	            }
        	}
        }
        current_content_sum += current_content[c_val];
    }

    return max_pos;
}

void set_sstd(struct sstd_data_c * sstd_data, struct shape_data_c * s_data, int sort) {
    straighten_log(STRAIGHTEN_VVINFO, "Generating the semistandard tableau for shape %s.", s_data[0].shape_string);  
    int32_t max_num_sstd = 10000;
    struct tableau * all_sstd_tableau = (struct tableau*) calloc(max_num_sstd, sizeof(struct tableau));
    set_tableau_bulk(all_sstd_tableau, max_num_sstd, s_data);

    struct tableau current_tableau = {};
    current_tableau.entries = (uint8_t*) calloc(s_data[0].num_boxes, sizeof(uint8_t));
    
    uint8_t * current_content = (uint8_t*) calloc(sstd_data[0].max_filling_value, sizeof(uint8_t));
    memcpy(current_content, sstd_data[0].content, sstd_data[0].max_filling_value*sizeof(uint8_t));

    int32_t num_sstd_tableau = set_sstd_threaded_helper(&all_sstd_tableau, &max_num_sstd, current_content, &current_tableau, sstd_data, s_data, 0, 0, s_data[0].num_boxes);
    
    if(sort) {
        uint8_t * all_sstd_tableau_entries_ptr = all_sstd_tableau[0].entries;
        // use some GCC tricks to make sure we are using the right qsort
        #ifdef _STRAIGHTEN_USE_QSORT_S
        qsort_s(all_sstd_tableau, num_sstd_tableau, sizeof(struct tableau), sstd_tableau_rowword_cmp_rev_s, s_data);
        #else
        qsort_r(all_sstd_tableau, num_sstd_tableau, sizeof(struct tableau), sstd_tableau_rowword_cmp_rev_r, s_data);
        #endif

        //this step is needed because copying to a new tableau array will ensure that the tableau entries are stored contiguously in memory
        struct tableau * all_sstd_tableau_final = (struct tableau*) calloc(num_sstd_tableau, sizeof(struct tableau));
        set_tableau_bulk(all_sstd_tableau_final, num_sstd_tableau, s_data);
        copy_tableau_bulk(all_sstd_tableau_final, all_sstd_tableau, num_sstd_tableau, s_data, 0);

        sstd_data[0].all_sstd_tableau = all_sstd_tableau_final;
        sstd_data[0].num_sstd_tableau = num_sstd_tableau;

        free(all_sstd_tableau_entries_ptr);
        free(all_sstd_tableau);
    }
    else {
        realloc_tableau_bulk_contiguous(&all_sstd_tableau, max_num_sstd, num_sstd_tableau, s_data);
        sstd_data[0].all_sstd_tableau = all_sstd_tableau;
        sstd_data[0].num_sstd_tableau = num_sstd_tableau;
    }    

    free(current_tableau.entries);
    free(current_content);

    straighten_log(STRAIGHTEN_INFO, "Total of %d semistandard tableau.", num_sstd_tableau);
}

void set_sstd_threaded(struct sstd_data_c * sstd_data, struct shape_data_c * s_data, int sort) {
    straighten_log(STRAIGHTEN_VVINFO, "Generating the semistandard tableau in threaded mode for shape %s.", s_data[0].shape_string);  
	int32_t max_num_seed = 10000;
    struct tableau seed_current_tableau = {};
    seed_current_tableau.entries = (uint8_t*) calloc(s_data[0].num_boxes, sizeof(uint8_t));
    
    uint8_t * seed_current_content = (uint8_t*) calloc(sstd_data[0].max_filling_value, sizeof(uint8_t));
    memcpy(seed_current_content, sstd_data[0].content, sstd_data[0].max_filling_value*sizeof(uint8_t));

    s_data[0].num_boxes = s_data[0].num_boxes + _STRAIGHTEN_CPU_CACHE_LINE_SIZE;
    struct tableau * seed = (struct tableau*) calloc(max_num_seed, sizeof(struct tableau));
    set_tableau_bulk(seed, max_num_seed, s_data);    

    int max_seed_col = 0;
    int max_seed_box = 0;
    int seed_num_entries = 0;
    while (seed_num_entries < _STRAIGHTEN_NUM_THREADS*100 && max_seed_col < s_data[0].shape[0]) {
    	max_seed_box += s_data[0].conjugate[max_seed_col];
    	max_seed_col++;
    	seed_num_entries = set_sstd_threaded_helper(&seed, &max_num_seed, seed_current_content, &seed_current_tableau, sstd_data, s_data, 0, 0, max_seed_box);
    }
    s_data[0].num_boxes = s_data[0].num_boxes - _STRAIGHTEN_CPU_CACHE_LINE_SIZE;

    struct tableau * all_sstd_tableau;
    int num_sstd_tableau = 0;
    if(max_seed_box == s_data[0].num_boxes) {
        num_sstd_tableau = seed_num_entries;
        all_sstd_tableau = (struct tableau*) calloc(num_sstd_tableau, sizeof(struct tableau));
        set_tableau_bulk(all_sstd_tableau, num_sstd_tableau, s_data);
        copy_tableau_bulk(all_sstd_tableau, seed, num_sstd_tableau, s_data, 0);
    }
    else {
        int32_t * max_num_sstd = (int32_t*) calloc(_STRAIGHTEN_NUM_THREADS, sizeof(int32_t));
        struct tableau ** all_sstd_tableau_threads = (struct tableau**) calloc(_STRAIGHTEN_NUM_THREADS, sizeof(struct tableau*));

        uint32_t * all_sstd_tableau_max_pos = (uint32_t*) calloc(_STRAIGHTEN_NUM_THREADS, sizeof(uint32_t));;
        uint8_t ** current_content = (uint8_t**) calloc(_STRAIGHTEN_NUM_THREADS, sizeof(uint8_t*)); 
         
        for(int thread = 0; thread < _STRAIGHTEN_NUM_THREADS; thread++) {
            max_num_sstd[thread] = 10000;
            all_sstd_tableau_threads[thread] = (struct tableau*) calloc(max_num_sstd[thread], sizeof(struct tableau));
            set_tableau_bulk(all_sstd_tableau_threads[thread], max_num_sstd[thread], s_data);
            current_content[thread] = (uint8_t*) calloc(sstd_data[0].max_filling_value + _STRAIGHTEN_CPU_CACHE_LINE_SIZE, sizeof(uint8_t));
        }

        int tid = 0;   
        #pragma omp parallel num_threads(_STRAIGHTEN_NUM_THREADS) private(tid)
        {
        	#pragma omp for schedule(dynamic)
            for(int tab=0; tab < seed_num_entries; tab++) {
            	tid = omp_get_thread_num();
            	memcpy(current_content[tid], sstd_data[0].content, sstd_data[0].max_filling_value*sizeof(uint8_t));
            	for(int b=0; b < max_seed_box; b++) {
            		current_content[tid][(seed[tab].entries[b] - 1)]--;
            	}
                all_sstd_tableau_max_pos[tid] = set_sstd_threaded_helper(all_sstd_tableau_threads + tid, max_num_sstd + tid, current_content[tid], seed + tab, sstd_data, s_data, max_seed_box, all_sstd_tableau_max_pos[tid], s_data[0].num_boxes);
            }
        }

        for(int thread = 0; thread < _STRAIGHTEN_NUM_THREADS; thread++) {
            num_sstd_tableau += all_sstd_tableau_max_pos[thread];
        }
        all_sstd_tableau = (struct tableau*) calloc(num_sstd_tableau, sizeof(struct tableau));
        set_tableau_bulk(all_sstd_tableau, num_sstd_tableau, s_data);
        int offset = 0;
        for(int thread = 0; thread < _STRAIGHTEN_NUM_THREADS; thread++) {
            copy_tableau_bulk(all_sstd_tableau + offset, all_sstd_tableau_threads[thread], all_sstd_tableau_max_pos[thread], s_data, 0);
            offset += all_sstd_tableau_max_pos[thread];

            free(all_sstd_tableau_threads[thread][0].entries);
            free(all_sstd_tableau_threads[thread]);            
            free(current_content[thread]);
        }

        free(max_num_sstd);
        free(all_sstd_tableau_max_pos);
        free(all_sstd_tableau_threads);
        free(current_content);
    }

    if(sort) {
        uint8_t * all_sstd_tableau_entries_ptr = all_sstd_tableau[0].entries;
        // use some GCC tricks to make sure we are using the right qsort
        #ifdef _STRAIGHTEN_USE_QSORT_S
        qsort_s(all_sstd_tableau, num_sstd_tableau, sizeof(struct tableau), sstd_tableau_rowword_cmp_rev_s, s_data);
        #else
        qsort_r(all_sstd_tableau, num_sstd_tableau, sizeof(struct tableau), sstd_tableau_rowword_cmp_rev_r, s_data);
        #endif

        //this step is needed because copying to a new tableau array will ensure that the tableau entries are stored contiguously in memory
        struct tableau * all_sstd_tableau_final = (struct tableau*) calloc(num_sstd_tableau, sizeof(struct tableau));
        set_tableau_bulk(all_sstd_tableau_final, num_sstd_tableau, s_data);
        copy_tableau_bulk(all_sstd_tableau_final, all_sstd_tableau, num_sstd_tableau, s_data, 0);

        sstd_data[0].all_sstd_tableau = all_sstd_tableau_final;
        sstd_data[0].num_sstd_tableau = num_sstd_tableau;

        free(all_sstd_tableau_entries_ptr);
        free(all_sstd_tableau);
    }
    else {
        sstd_data[0].all_sstd_tableau = all_sstd_tableau;
        sstd_data[0].num_sstd_tableau = num_sstd_tableau;
    }    

    free(seed[0].entries);
    free(seed);
    free(seed_current_content);
    free(seed_current_tableau.entries);

    straighten_log(STRAIGHTEN_INFO, "Total of %d semistandard tableau.", num_sstd_tableau);
}


void set_tableaux_rowcontent(struct sstd_data_c * sstd_data, struct shape_data_c * s_data) {
	uint8_t * sstd_row_content = (uint8_t*) calloc(sstd_data[0].num_sstd_tableau * s_data[0].conjugate[0] * sstd_data[0].max_filling_value, sizeof(uint8_t));
	sstd_data[0].sstd_row_content = sstd_row_content;
	uint32_t box, c_val;

	for(int tab = 0; tab < sstd_data[0].num_sstd_tableau; tab++) {
		for(int row = 0; row < s_data[0].conjugate[0]; row++) {
			for(int col = 0; col < s_data[0].shape[row]; col++) {
				box = s_data[0].row_col_to_box[row*s_data[0].shape[0] + col];
				c_val = sstd_data[0].all_sstd_tableau[tab].entries[box] - 1;
				sstd_row_content[row * sstd_data[0].max_filling_value + c_val]++;
			}
		}
		sstd_row_content += s_data[0].conjugate[0] * sstd_data[0].max_filling_value;
	}
}

void set_D_basis_matrix(struct sstd_data_c * sstd_data, struct shape_data_c * s_data) {
    straighten_log(STRAIGHTEN_VVINFO, "Generating the D-basis sparse matrix for shape %s.", s_data[0].shape_string);  
	struct sparse_CSR_32 * D_basis_matrix = (struct sparse_CSR_32*) calloc(1, sizeof(struct sparse_CSR_32));
	sstd_data[0].D_basis_matrix = D_basis_matrix;
    
    D_basis_matrix[0].number_rows = sstd_data[0].num_sstd_tableau;
    D_basis_matrix[0].number_columns = sstd_data[0].num_sstd_tableau;
    D_basis_matrix[0].rows_ptr = (uint32_t*) calloc(sstd_data[0].num_sstd_tableau+1, sizeof(uint32_t));
    D_basis_matrix[0].values = (int32_t*) calloc(sstd_data[0].num_sstd_tableau*100, sizeof(int32_t));
    D_basis_matrix[0].columns_index = (uint32_t*) calloc(sstd_data[0].num_sstd_tableau*100, sizeof(uint32_t));
    D_basis_matrix[0].rows_ptr_length = sstd_data[0].num_sstd_tableau+1;
    D_basis_matrix[0].values_length = sstd_data[0].num_sstd_tableau*100;
    
    uint8_t * t_data_c = (uint8_t*) calloc(s_data[0].conjugate[0] * sstd_data[0].max_filling_value, sizeof(uint8_t));
    int64_t * temp_row_vals = (int64_t*) calloc(sstd_data[0].num_sstd_tableau, sizeof(int64_t));
    
    int64_t coeff;
    uint32_t temp_index;
    for(int tab = 0; tab < sstd_data[0].num_sstd_tableau; tab++) {
    	memset(temp_row_vals, 0, sstd_data[0].num_sstd_tableau*sizeof(int64_t));
        temp_row_vals[tab] = 1;
        for(int lower_tab = 0; lower_tab < tab; lower_tab++) {
        	coeff = c_compute_rearrangement_coeff(sstd_data[0].all_sstd_tableau[tab].entries, lower_tab, t_data_c, sstd_data, s_data);     
        	if(coeff != 0) {
        		for(int pos=D_basis_matrix[0].rows_ptr[lower_tab]; pos < D_basis_matrix[0].rows_ptr[lower_tab+1]; pos++) {
        			temp_row_vals[D_basis_matrix[0].columns_index[pos]] = temp_row_vals[D_basis_matrix[0].columns_index[pos]] - (coeff * D_basis_matrix[0].values[pos]);
        		}
        	}
        }

        temp_index = D_basis_matrix[0].rows_ptr[tab];
        for(int lower_tab = 0; lower_tab <= tab; lower_tab++) {
        	if(temp_row_vals[lower_tab] != 0) {
        		if(temp_row_vals[lower_tab] < INT_MIN || temp_row_vals[lower_tab] > INT_MAX) {
                    straighten_log(STRAIGHTEN_FATAL, "D-Basis coefficient too big to fit in an INT32.");
                    exit(EXIT_FAILURE);
                }
                D_basis_matrix[0].values[temp_index] = temp_row_vals[lower_tab];
                D_basis_matrix[0].columns_index[temp_index] = lower_tab;
                temp_index += 1;

                if(temp_index >= D_basis_matrix[0].values_length) {
	                straighten_log(STRAIGHTEN_VINFO, "Enlarging the size of the D-Basis sparse matrix.");
	                D_basis_matrix[0].values = (int32_t*)realloc(D_basis_matrix[0].values, D_basis_matrix[0].values_length * 2 * sizeof(int32_t));
	                D_basis_matrix[0].columns_index = (uint32_t*)realloc(D_basis_matrix[0].columns_index, D_basis_matrix[0].values_length * 2 * sizeof(uint32_t));
	                D_basis_matrix[0].values_length = D_basis_matrix[0].values_length * 2;
	            }
        	}
        }
        D_basis_matrix[0].rows_ptr[tab+1] = temp_index;
    }

    D_basis_matrix[0].values = (int32_t*) realloc(D_basis_matrix[0].values,D_basis_matrix[0].rows_ptr[sstd_data[0].num_sstd_tableau] * sizeof(int32_t));
    D_basis_matrix[0].columns_index = (uint32_t*) realloc(D_basis_matrix[0].columns_index, D_basis_matrix[0].rows_ptr[sstd_data[0].num_sstd_tableau] * sizeof(uint32_t));
    D_basis_matrix[0].values_length = D_basis_matrix[0].rows_ptr[sstd_data[0].num_sstd_tableau];
    
    free(t_data_c);
}

void set_R_coeff_skip_matrix(struct sstd_data_c * sstd_data, struct shape_data_c * s_data) {  
    straighten_log(STRAIGHTEN_VVINFO, "Generating the R-Skip coefficient sparse matrix for %s.", s_data[0].shape_string);  
    struct sparse_CSR_8 *  R_coeff_skip = (struct sparse_CSR_8*) calloc(1, sizeof(struct sparse_CSR_8));
    sstd_data[0].R_coeff_skip = R_coeff_skip;

    R_coeff_skip[0].number_rows = sstd_data[0].num_sstd_tableau;
    R_coeff_skip[0].number_columns = sstd_data[0].num_sstd_tableau;
    R_coeff_skip[0].rows_ptr = (uint32_t*) calloc(sstd_data[0].num_sstd_tableau+1, sizeof(uint32_t));
    R_coeff_skip[0].columns_index = (uint32_t*) calloc(sstd_data[0].num_sstd_tableau*100, sizeof(uint32_t));
    R_coeff_skip[0].rows_ptr_length = sstd_data[0].num_sstd_tableau+1;
    R_coeff_skip[0].values_length = sstd_data[0].num_sstd_tableau*100;

    int temp_index;
    for(int tab=0; tab < sstd_data[0].num_sstd_tableau; tab++) {
    	temp_index = R_coeff_skip[0].rows_ptr[tab];

    	for(int lower_tab=0; lower_tab <= tab; lower_tab++) {
    		if(sstd_tableau_colword_cmp_s(s_data, sstd_data[0].all_sstd_tableau + lower_tab,sstd_data[0].all_sstd_tableau + tab) <= 0) {
    			R_coeff_skip[0].columns_index[temp_index] = lower_tab;
                temp_index += 1;

                if(temp_index >= R_coeff_skip[0].values_length) {
                    straighten_log(STRAIGHTEN_VINFO, "Enlarging the size of the R-Skip sparse matrix.");
                    R_coeff_skip[0].columns_index = (uint32_t*)realloc(R_coeff_skip[0].columns_index, R_coeff_skip[0].values_length * 2 * sizeof(uint32_t));
                    R_coeff_skip[0].values_length = R_coeff_skip[0].values_length * 2;
                }
    		}
    	}
    	R_coeff_skip[0].rows_ptr[tab+1] = temp_index;
    }

    R_coeff_skip[0].columns_index = (uint32_t*)realloc(R_coeff_skip[0].columns_index, R_coeff_skip[0].rows_ptr[sstd_data[0].num_sstd_tableau] * sizeof(uint32_t));
    R_coeff_skip[0].values_length = R_coeff_skip[0].rows_ptr[sstd_data[0].num_sstd_tableau];
}


int set_dictionary_packed_seed(struct tableau ** tableau_array, int32_t * max_num_dictionary, uint8_t * current_content, struct tableau * current_tableau, struct sstd_data_c * sstd_data, struct shape_data_c * s_data, int box, int max_pos, int max_box) {
	if (box == max_box) {
        for(int b = 0; b < max_box; b++) {
            tableau_array[0][max_pos].entries[b] = current_tableau[0].entries[b];
        }
        max_pos++;
        if(max_pos == *max_num_dictionary) {
            //need to enlarge the storage tableau array
            int32_t old_size = *max_num_dictionary;
            *max_num_dictionary = (*max_num_dictionary)*2;
            realloc_tableau_bulk_contiguous(tableau_array, old_size, *max_num_dictionary, s_data);            
        }
        return max_pos;
    }

    int col = s_data[0].box_to_col[box];
    int row = s_data[0].box_to_row[box];

    int num_boxes_right = 0;
    for(int c = col+1; c < s_data[0].shape[0]; c++) {
    	num_boxes_right += s_data[0].conjugate[c];
    }

    int current_content_sum = 0;
    int c_less = 0;
    for(int c_val = 0; c_val < sstd_data[0].max_filling_value; c_val++) {
    	// this checks that adding the current content value would not exceed the maximum number of this value allowed
    	if(current_content[c_val] != 0) {
    		// this checks that the number of boxes that are strictly to the right can fit the content values less than the current box
    		if(num_boxes_right >= current_content_sum) {
    			// this checks that the current content value is not to great to allow the column to be filled
	        	if (((c_val + 1) + (s_data[0].conjugate[col] - row - 1)) <= sstd_data[0].max_filling_value) {
	        		if(col != 0) {
                        c_less = compare_uint8_lex(current_tableau[0].entries + s_data[0].row_col_to_box[col-1], current_tableau[0].entries + s_data[0].row_col_to_box[col], row);
	        		}
                    
	        		// this checks that the values will be weakly increasing in rows
	            	if(col == 0  || ((c_less < 0) || ((s_data[0].conjugate[col-1] != s_data[0].conjugate[col]) || (current_tableau[0].entries[s_data[0].col_row_to_box[((col-1)*s_data[0].shape_length) + row]] <= c_val + 1)))) {
	            		// this checks that values will be strictly increasing in columns
	                	if (row==0 || current_tableau[0].entries[box-1] < c_val + 1) {
	                		// the current content value passes all the checks, recurse to the next box
	                    	current_content[c_val] -= 1;
	                        current_tableau[0].entries[box] = c_val + 1;
	                        max_pos = set_dictionary_packed_seed(tableau_array, max_num_dictionary, current_content, current_tableau, sstd_data, s_data, box+1, max_pos, max_box);
	                        current_content[c_val] += 1;
	                    }
	                }
	            }
        	}
        }
        current_content_sum += current_content[c_val];
    }

    return max_pos;
}

int set_dictionary_packed_helper(struct packed_tableau ** tableau_array, int32_t * max_num_dictionary, uint8_t * current_content, uint8_t * work, struct tableau * current_tableau, struct sstd_data_c * sstd_data, struct shape_data_c * s_data, int box, int max_pos) {
	if(box == s_data[0].box_start_single_rows) {
		int c_val;
        int c_less = 0;
        int c_thresh = 0;
        for(int b=s_data[0].box_start_single_rows; b < s_data[0].num_boxes; b++) {
        	c_val = 0;
        	c_thresh = current_content[c_val];
        	while(current_content[c_val] == 0 || c_less >= c_thresh) {
        		c_val++;
        		c_thresh += current_content[c_val];
        	}
        	c_less++;
        	current_tableau[0].entries[b] = c_val + 1;
        }

        if(sstd_data[0].lower_bound_sstd == 0 || (get_standardization_of_colstraightened(current_tableau, sstd_data, s_data, work) >= sstd_data[0].lower_bound_sstd)) {
	        set_packed_mult(tableau_array[0][max_pos].entries, current_tableau[0].entries, s_data[0].num_boxes);
            max_pos++;
            if(max_pos == *max_num_dictionary) {
                //need to enlarge the storage tableau array
                int32_t old_size = *max_num_dictionary;
                *max_num_dictionary = (*max_num_dictionary)*2;
                realloc_packed_tableau_bulk_contiguous(tableau_array, old_size, *max_num_dictionary, s_data);                
            }
	    	return max_pos;
    	}
    	else {
    		return max_pos;
    	}
	}

    int col = s_data[0].box_to_col[box];
    int row = s_data[0].box_to_row[box];

    int num_boxes_right = 0;
    for(int c = col+1; c < s_data[0].shape[0]; c++) {
    	num_boxes_right += s_data[0].conjugate[c];
    }

    int current_content_sum = 0;
    int c_less = 0;
    for(int c_val = 0; c_val < sstd_data[0].max_filling_value; c_val++) {
    	// this checks that adding the current content value would not exceed the maximum number of this value allowed
    	if(current_content[c_val] != 0) {
    		// this checks that the number of boxes that are strictly to the right can fit the content values less than the current box
    		if(num_boxes_right >= current_content_sum) {
    			// this checks that the current content value is not to great to allow the column to be filled
	        	if (((c_val + 1) + (s_data[0].conjugate[col] - row - 1)) <= sstd_data[0].max_filling_value) {
	        		if(col != 0) {
                        c_less = compare_uint8_lex(current_tableau[0].entries + s_data[0].row_col_to_box[col-1], current_tableau[0].entries + s_data[0].row_col_to_box[col], row);
	        		}
                    
	        		// this checks that the values will be weakly increasing in rows
	            	if(col == 0  || ((c_less < 0) || ((s_data[0].conjugate[col-1] != s_data[0].conjugate[col]) || (current_tableau[0].entries[s_data[0].col_row_to_box[((col-1)*s_data[0].shape_length) + row]] <= c_val + 1)))) {
	            		// this checks that values will be strictly increasing in columns
	                	if (row==0 || current_tableau[0].entries[box-1] < c_val + 1) {
	                		// the current content value passes all the checks, recurse to the next box
	                    	current_content[c_val] -= 1;
	                        current_tableau[0].entries[box] = c_val + 1;
	                        max_pos = set_dictionary_packed_helper(tableau_array, max_num_dictionary, current_content, work, current_tableau, sstd_data, s_data, box+1, max_pos);
	                        current_content[c_val] += 1;
	                    }
	                }
	            }
        	}
        }
        current_content_sum += current_content[c_val];
    }

    return max_pos;
}

void set_dictionary_packed_threaded(struct sstd_data_c * sstd_data, struct shape_data_c * s_data) {
	straighten_log(STRAIGHTEN_VVINFO, "Generating dictionary tableau for %s.", s_data[0].shape_string);
    int32_t max_num_seed = 10000;
    struct tableau seed_current_tableau = {};
    seed_current_tableau.entries = (uint8_t*) calloc(s_data[0].num_boxes, sizeof(uint8_t));
    
    uint8_t * seed_current_content = (uint8_t*) calloc(sstd_data[0].max_filling_value, sizeof(uint8_t));
    memcpy(seed_current_content, sstd_data[0].content, sstd_data[0].max_filling_value*sizeof(uint8_t));

    s_data[0].num_boxes = s_data[0].num_boxes + _STRAIGHTEN_CPU_CACHE_LINE_SIZE;
    struct tableau * seed = (struct tableau*) calloc(max_num_seed, sizeof(struct tableau));
    set_tableau_bulk(seed, max_num_seed, s_data);

    int max_seed_col = 0;
    int max_seed_box = 0;
    int seed_num_entries = 0;
    if(s_data[0].col_start_single_rows > 0) {
        while (seed_num_entries < _STRAIGHTEN_NUM_THREADS*50 && max_seed_col < s_data[0].col_start_single_rows) {
        	max_seed_box += s_data[0].conjugate[max_seed_col];
        	max_seed_col++;
        	seed_num_entries = set_dictionary_packed_seed(&seed, &max_num_seed, seed_current_content, &seed_current_tableau, sstd_data, s_data, 0, 0, max_seed_box);
        }
    }
    else {
        seed_num_entries = 1;
    }
    s_data[0].num_boxes = s_data[0].num_boxes - _STRAIGHTEN_CPU_CACHE_LINE_SIZE;


    straighten_log(STRAIGHTEN_VINFO, "Total of %d seed dictionary tableau for boxes up to %d.", seed_num_entries, max_seed_box);
    
    int32_t * max_num_dictionary_thread = (int32_t*) calloc(_STRAIGHTEN_NUM_THREADS, sizeof(int32_t));
    struct packed_tableau ** all_dictionary_tableau_threads = (struct packed_tableau**) calloc(_STRAIGHTEN_NUM_THREADS, sizeof(struct packed_tableau*));

    uint32_t * all_dictionary_tableau_max_pos = (uint32_t*) calloc(_STRAIGHTEN_NUM_THREADS, sizeof(uint32_t));;
    uint8_t ** current_content = (uint8_t**) calloc(_STRAIGHTEN_NUM_THREADS, sizeof(uint8_t*));
    uint8_t ** work = (uint8_t**) calloc(_STRAIGHTEN_NUM_THREADS, sizeof(uint8_t*));
     
    for(int thread = 0; thread < _STRAIGHTEN_NUM_THREADS; thread++) {
        max_num_dictionary_thread[thread] = 100000;
        all_dictionary_tableau_threads[thread] = (struct packed_tableau*) calloc(max_num_dictionary_thread[thread], sizeof(struct packed_tableau));
        set_packed_tableau_bulk(all_dictionary_tableau_threads[thread], max_num_dictionary_thread[thread], s_data);
        work[thread] = (uint8_t*) calloc(s_data[0].num_boxes + _STRAIGHTEN_CPU_CACHE_LINE_SIZE, sizeof(uint8_t));
        current_content[thread] = (uint8_t*) calloc(sstd_data[0].max_filling_value + _STRAIGHTEN_CPU_CACHE_LINE_SIZE, sizeof(uint8_t));
    }

    int32_t max_dictionary_tableau = 1000000;
    struct packed_tableau * all_dictionary_tableau = (struct packed_tableau*) calloc(max_dictionary_tableau, sizeof(struct packed_tableau));
    set_packed_tableau_bulk(all_dictionary_tableau, max_dictionary_tableau, s_data);
    uint32_t num_dictionary_tableau = 0;

    uint32_t iterates = seed_num_entries/_STRAIGHTEN_NUM_THREADS + ((seed_num_entries % _STRAIGHTEN_NUM_THREADS) != 0);
    uint32_t threshhold = 10000000;
    uint32_t offset = 0;
    int32_t * offseter = (int32_t*) calloc(_STRAIGHTEN_NUM_THREADS, sizeof(int32_t));
    for(int chunk=0; chunk < iterates; chunk++) {
        #pragma omp parallel num_threads(_STRAIGHTEN_NUM_THREADS)
    	{
			#pragma omp for schedule(static)
			for(int thread=0; thread < _STRAIGHTEN_NUM_THREADS; thread++) {
				if(((_STRAIGHTEN_NUM_THREADS * chunk)+thread) < seed_num_entries) {
	    			memcpy(current_content[thread], sstd_data[0].content, sstd_data[0].max_filling_value*sizeof(uint8_t));
	    			for(int pos = 0; pos < max_seed_box; pos++) {
	                    current_content[thread][seed[(_STRAIGHTEN_NUM_THREADS * chunk)+thread].entries[pos]-1]--;
	    			}
	    			all_dictionary_tableau_max_pos[thread] = set_dictionary_packed_helper(all_dictionary_tableau_threads + thread, max_num_dictionary_thread + thread, current_content[thread], work[thread], seed + (_STRAIGHTEN_NUM_THREADS * chunk)+thread, sstd_data, s_data, max_seed_box, all_dictionary_tableau_max_pos[thread]);
	    		}
			}
		}
        
        offset = 0;
        for(int thread=0; thread < _STRAIGHTEN_NUM_THREADS; thread++) {
            offseter[thread] = offset;
            offset += all_dictionary_tableau_max_pos[thread];
        }

        if((num_dictionary_tableau + offset) > _STRAIGHTEN_MAX_DICTIONARY) {
            straighten_log(STRAIGHTEN_WARNING, "Maximum size exceeded, returning a subset of the dictionary tableau of size %d. \n", num_dictionary_tableau);
            //this breaks out of the two four loops
            chunk = iterates;
        }
        else {
            if((num_dictionary_tableau + offset) > max_dictionary_tableau) {
                int old_size = max_dictionary_tableau;
                while((num_dictionary_tableau + offset) > max_dictionary_tableau) {
                    max_dictionary_tableau = max_dictionary_tableau*2;
                }
                realloc_packed_tableau_bulk_contiguous(&all_dictionary_tableau, old_size, max_dictionary_tableau, s_data);
            }

            #pragma omp parallel num_threads(_STRAIGHTEN_NUM_THREADS)
            {
                #pragma omp for schedule(static)   
                for(int thread=0; thread < _STRAIGHTEN_NUM_THREADS; thread++) {
                    copy_packed_tableau_bulk(all_dictionary_tableau + num_dictionary_tableau + offseter[thread], all_dictionary_tableau_threads[thread], all_dictionary_tableau_max_pos[thread], s_data, 1);
                    all_dictionary_tableau_max_pos[thread] = 0;
                }
            }
            num_dictionary_tableau += offset;
        }
        
        /*for(int thread=0; thread < _STRAIGHTEN_NUM_THREADS; thread++) {
			if(all_dictionary_tableau_max_pos[thread] > 0) {
				if((num_dictionary_tableau + all_dictionary_tableau_max_pos[thread]) > _STRAIGHTEN_MAX_DICTIONARY) {
		            straighten_log(STRAIGHTEN_WARNING, "Maximum size exceeded, returning a subset of the dictionary tableau of size %d. \n", num_dictionary_tableau);
                    //this breaks out of the two four loops
                    chunk = iterates;
                    thread = _STRAIGHTEN_NUM_THREADS;
		        }
                else {
                    if((num_dictionary_tableau + all_dictionary_tableau_max_pos[thread]) > max_dictionary_tableau) {
                        while((num_dictionary_tableau + all_dictionary_tableau_max_pos[thread]) > max_dictionary_tableau) {
                            max_dictionary_tableau = max_dictionary_tableau*2;
                        }
                        realloc_packed_tableau_bulk_contiguous(&all_dictionary_tableau, max_dictionary_tableau, s_data);
                    }
                    copy_packed_tableau_bulk(all_dictionary_tableau + num_dictionary_tableau, all_dictionary_tableau_threads[thread], all_dictionary_tableau_max_pos[thread], s_data, 1);
                    num_dictionary_tableau += all_dictionary_tableau_max_pos[thread]; 
                    all_dictionary_tableau_max_pos[thread] = 0;
                }
			}			   			
		}*/
		
    	if(num_dictionary_tableau > threshhold) {
            threshhold += 10000000;
            straighten_log(STRAIGHTEN_VINFO, "Total of %d dictionary tableau processed so far.", num_dictionary_tableau);
        }
    }

    realloc_packed_tableau_bulk_contiguous(&all_dictionary_tableau, max_dictionary_tableau, num_dictionary_tableau, s_data);
    sstd_data[0].all_dictionary_tableau = all_dictionary_tableau;
	sstd_data[0].num_dictionary_tableau = num_dictionary_tableau;
    
    free(seed[0].entries);
    free(seed);
    free(seed_current_tableau.entries);
    free(seed_current_content);
    
    for(int thread=0; thread < _STRAIGHTEN_NUM_THREADS; thread++) {
        free(current_content[thread]);
        free(work[thread]);
        free(all_dictionary_tableau_threads[thread][0].entries);
        free(all_dictionary_tableau_threads[thread]);
    }
    free(all_dictionary_tableau_threads);
    free(all_dictionary_tableau_max_pos);
    free(max_num_dictionary_thread);
    free(current_content);
    free(work);
    free(offseter);

    straighten_log(STRAIGHTEN_INFO, "Total of %d dictionary tableau.", num_dictionary_tableau);
}

void set_straightening_cache(struct sstd_data_c * sstd_data) {
	sstd_data[0].straighten_cache_index_one = (uint8_t*) calloc(sstd_data[0].num_dictionary_tableau, sizeof(uint8_t));
    sstd_data[0].straighten_cache_index_two = (uint32_t*) calloc(sstd_data[0].num_dictionary_tableau, sizeof(uint32_t));
    for(int tab=0; tab < sstd_data[0].num_dictionary_tableau; tab++) {
    	sstd_data[0].straighten_cache_index_one[tab] = UINT8_MAX;
    }

    sstd_data[0].straighten_cache = (struct sparse_cache_CSR_32*) calloc(_STRAIGHTEN_NUM_THREADS, sizeof(struct sparse_cache_CSR_32));

    for(int thread = 0; thread < _STRAIGHTEN_NUM_THREADS; thread++) {
        if(sstd_data[0].num_dictionary_tableau > 0) {
        	sstd_data[0].straighten_cache[thread].values = (int16_t*) calloc(_STRAIGHTEN_PERTHREAD_CACHE, sizeof(int16_t));
            sstd_data[0].straighten_cache[thread].index = (uint16_t*) calloc(_STRAIGHTEN_PERTHREAD_CACHE, sizeof(uint16_t));
            sstd_data[0].straighten_cache[thread].rows_ptr = (uint32_t*) calloc(_STRAIGHTEN_PERTHREAD_CACHE_ENTRIES, sizeof(uint32_t));
        }
        sstd_data[0].straighten_cache[thread].rows_ptr_max_length = _STRAIGHTEN_PERTHREAD_CACHE_ENTRIES;
        sstd_data[0].straighten_cache[thread].filled_flag = 0;
        sstd_data[0].straighten_cache[thread].values_length = _STRAIGHTEN_PERTHREAD_CACHE;
        sstd_data[0].straighten_cache[thread].rows_ptr_current = 0;
    }
    
}

void clear_straightening_cache(struct sstd_data_c * sstd_data, int32_t zero) {
    if(sstd_data[0].num_dictionary_tableau > 0) {
        memset(sstd_data[0].straighten_cache_index_two, 0, sstd_data[0].num_dictionary_tableau * sizeof(uint32_t));
        for(int tab=0; tab < sstd_data[0].num_dictionary_tableau; tab++) {
            sstd_data[0].straighten_cache_index_one[tab] = UINT8_MAX;
        }

        for(int thread = 0; thread < _STRAIGHTEN_NUM_THREADS; thread++) {
            if(zero) {
                memset(sstd_data[0].straighten_cache[thread].values, 0, sstd_data[0].straighten_cache[thread].values_length * sizeof(int16_t));
                memset(sstd_data[0].straighten_cache[thread].index, 0, sstd_data[0].straighten_cache[thread].values_length * sizeof(uint16_t));
                memset(sstd_data[0].straighten_cache[thread].rows_ptr, 0, sstd_data[0].straighten_cache[thread].rows_ptr_max_length * sizeof(uint32_t));
            }
            sstd_data[0].straighten_cache[thread].filled_flag = 0;
            sstd_data[0].straighten_cache[thread].rows_ptr_current = 0;
        }
    }
}

void set_sstd_column_factorial(struct sstd_data_c * sstd_data, struct shape_data_c * s_data) {
    sstd_data[0].sstd_column_factorial = (uint64_t *) calloc(sstd_data[0].num_sstd_tableau, sizeof(uint64_t));
    uint64_t fac;
    int repeat_num;
    uint8_t * prev_col_ptr;
    uint8_t * curr_col_ptr;
    int curr_col;
    for(int tab = 0; tab < sstd_data[0].num_sstd_tableau; tab++) {
        fac = 1;
        for(int u_col = 0; u_col < s_data[0].unique_columns[0].length; u_col++) {
            repeat_num = 1;
            for(int c = 1; c < s_data[0].unique_columns[0].unique_column_number[u_col]; c++) {
                curr_col = s_data[0].unique_columns[0].unique_column_start[u_col] + c;
                prev_col_ptr = sstd_data[0].all_sstd_tableau[tab].entries + s_data[0].row_col_to_box[curr_col-1];
                curr_col_ptr = sstd_data[0].all_sstd_tableau[tab].entries + s_data[0].row_col_to_box[curr_col];
                if(compare_uint8_lex(prev_col_ptr, curr_col_ptr, s_data[0].conjugate[curr_col]) == 0) {
                    repeat_num++;
                }
                else {
                    fac *= factorial(repeat_num);
                    repeat_num = 1;
                }
            }
            fac *= factorial(repeat_num);
        }
        sstd_data[0].sstd_column_factorial[tab] = fac;
    }
}

void set_sstd_column_repeat_data(struct sstd_data_c * sstd_data, struct shape_data_c * s_data, uint32_t sstd_index, uint32_t ** column_repeats) {
    column_repeats[0] = (uint32_t *) calloc(sstd_data[0].max_filling_value * s_data[0].unique_columns[0].length, sizeof(uint32_t));
    int repeat_num;
    uint8_t * prev_col_ptr;
    uint8_t * curr_col_ptr;
    int curr_col;
    for(int u_col = 0; u_col < s_data[0].unique_columns[0].length; u_col++) {
        repeat_num = 1;
        for(int c = 1; c < s_data[0].unique_columns[0].unique_column_number[u_col]; c++) {
            curr_col = s_data[0].unique_columns[0].unique_column_start[u_col] + c;
            prev_col_ptr = sstd_data[0].all_sstd_tableau[sstd_index].entries + s_data[0].row_col_to_box[curr_col-1];
            curr_col_ptr = sstd_data[0].all_sstd_tableau[sstd_index].entries + s_data[0].row_col_to_box[curr_col];
            if(compare_uint8_lex(prev_col_ptr, curr_col_ptr, s_data[0].conjugate[curr_col]) == 0) {
                repeat_num++;
            }
            else {
                column_repeats[0][(u_col * sstd_data[0].max_filling_value) + (repeat_num-1)]++;
                repeat_num = 1;
            }
        }
        column_repeats[0][(u_col * sstd_data[0].max_filling_value) + (repeat_num-1)]++;
    }
}

void set_sstd_column_overlap_data(struct sstd_data_c * sstd_data, struct shape_data_c * s_data, uint32_t sstd_index, uint32_t ** column_overlaps) {
    column_overlaps[0] = (uint32_t *) calloc(sstd_data[0].max_filling_value, sizeof(uint32_t));
    int first_val_in_col;
    int second_val_in_col;
    uint8_t val;
    int num_overlap;
    for(int c_val1 = 0; c_val1 < sstd_data[0].max_filling_value; c_val1++) {
        for(int c_val2 = c_val1+1; c_val2 < sstd_data[0].max_filling_value; c_val2++) {
            num_overlap = 0;
            for(int col = 0; col < s_data[0].shape[0]; col++) {
                first_val_in_col = 0;
                second_val_in_col = 0;
                for(int row = 0; row < s_data[0].conjugate[col]; row++) {
                    val = sstd_data[0].all_sstd_tableau[sstd_index].entries[s_data[0].row_col_to_box[row * s_data[0].shape[0] + col]]-1;
                    if(val == c_val1) {
                        first_val_in_col = 1;
                    }
                    else if(val == c_val2) {
                        second_val_in_col = 1;
                    }
                }
                if(first_val_in_col == 1 && second_val_in_col == 1) {
                    num_overlap++;
                }
            }
        if(num_overlap > 0) {
            column_overlaps[0][num_overlap-1]++;
        }           
        }        
    }
}

void set_sstd_row_contents_uid(struct sstd_data_c * sstd_data, struct shape_data_c * s_data, uint32_t sstd_index, uint8_t ** row_contents_uid) {
    row_contents_uid[0] = (uint8_t *) calloc((s_data[0].conjugate[0]) * sstd_data[0].max_filling_value, sizeof(uint8_t));
    uint8_t * temp = (uint8_t *) calloc((s_data[0].conjugate[0]) * sstd_data[0].max_filling_value, sizeof(uint8_t));
    uint8_t val;
    for(int row = 0; row < s_data[0].conjugate[0]; row++) {
        for(int col = 0; col < s_data[0].shape[row]; col++) {
            val = sstd_data[0].all_sstd_tableau[sstd_index].entries[s_data[0].row_col_to_box[row * s_data[0].shape[0] + col]]-1;
            //row_contents_uid[0][((row-1) * sstd_data[0].max_filling_value) + val]++;
            temp[(val * (s_data[0].conjugate[0])) + row]++;
        }
    }
    sort_fast_array(temp, s_data[0].conjugate[0], sstd_data[0].max_filling_value);

    for(int row = 0; row < (s_data[0].conjugate[0]); row++) {
        for(int col = 0; col < sstd_data[0].max_filling_value; col++) {
            row_contents_uid[0][(row * sstd_data[0].max_filling_value) + col] = temp[((sstd_data[0].max_filling_value - col - 1) * (s_data[0].conjugate[0])) + row];
        }
    }
    free(temp);
}

void construct_sstd_data_c(struct sstd_data_c * sstd_data, uint8_t * content, uint8_t max_filling_value, struct shape_data_c * s_data, struct sstd_data_c_options * options) {
	sstd_data[0].content = (uint8_t*) calloc(max_filling_value, sizeof(uint8_t));
	sstd_data[0].max_filling_value = max_filling_value;
	for(int c_val = 0; c_val < sstd_data[0].max_filling_value; c_val++) {
		sstd_data[0].content[c_val] = content[c_val];
	}

    uint32_t fac = factorial(max_filling_value);
    uint8_t * id = (uint8_t*) calloc(max_filling_value, sizeof(uint8_t));
    sstd_data[0].permutations = (uint8_t*) calloc(max_filling_value * fac, sizeof(uint8_t));
    
    for(int val = 0; val < max_filling_value; val++) {
        id[val]=val;
    }

    // this stores all permutations of id in sstd_data[0].permutations
    straighten_gen_permutations(id, max_filling_value, sstd_data[0].permutations);
    free(id);

    sstd_data[0].permutation_counts = (uint64_t*) calloc(max_filling_value+1, sizeof(uint64_t));
    sstd_data[0].permutation_counts[0] = 1;
    for(int val = 1; val  <= max_filling_value; val++) {
        sstd_data[0].permutation_counts[val] = val * sstd_data[0].permutation_counts[val-1];
    }

    if(options[0].set_sstd == 1) {
        if(options[0].gen_tableau_threaded == 0) {
	        straighten_timing_macro(set_sstd(sstd_data, s_data, !(options[0].do_not_sort_sstd)), "Generating the semistandard tableau");
        }
        else {
            straighten_timing_macro(set_sstd_threaded(sstd_data, s_data, !(options[0].do_not_sort_sstd)), "Generating the semistandard tableau");
        }
    }
    if(options[0].set_sstd_column_factorial == 1) {
        set_sstd_column_factorial(sstd_data, s_data);
    }
    if(options[0].set_rowcontent == 1 && options[0].set_Dbasis == 0) {
        set_tableaux_rowcontent(sstd_data, s_data);
    }
    if(options[0].set_Dbasis == 1) {
        set_tableaux_rowcontent(sstd_data, s_data);
        set_D_basis_matrix(sstd_data, s_data);
        set_R_coeff_skip_matrix(sstd_data, s_data);
    }
    if(options[0].use_lower_bound == 1) {
        sstd_data[0].lower_bound_sstd = (uint32_t)(options[0].lower_bound_fraction * (double)sstd_data[0].num_sstd_tableau);
    }
    if(options[0].set_dictionary == 1) {
	    straighten_timing_macro(set_dictionary_packed_threaded(sstd_data, s_data), "Generating the dicitionary tableau");
	}
    if(options[0].set_dictionary_hash == 1) {
        straighten_timing_macro(sstd_data[0].dictionary_hash = pt_build_hashtable(sstd_data[0].all_dictionary_tableau, sstd_data[0].num_dictionary_tableau, s_data[0].box_start_single_rows, s_data, options[0].dictionary_hash_copy_data), "Generating the dictionary tableau hash table");
	}
    if(options[0].set_cache == 1) {
        set_straightening_cache(sstd_data);
    }
}

void destruct_sstd_data_c(struct sstd_data_c * sstd_data) {
    if(sstd_data[0].content != NULL) {
        free(sstd_data[0].content);
        sstd_data[0].content = NULL;
    }
    if(sstd_data[0].permutations != NULL) {
        free(sstd_data[0].permutations);
        sstd_data[0].permutations = NULL;
    }
    if(sstd_data[0].permutation_counts != NULL) {
        free(sstd_data[0].permutation_counts);
        sstd_data[0].permutation_counts = NULL;
    }
    if(sstd_data[0].all_sstd_tableau != NULL) {
        free(sstd_data[0].all_sstd_tableau[0].entries);
        free(sstd_data[0].all_sstd_tableau);
        sstd_data[0].all_sstd_tableau = NULL;
    }
    if(sstd_data[0].sstd_row_content != NULL) {
        free(sstd_data[0].sstd_row_content);
        sstd_data[0].sstd_row_content = NULL;
    }
    if(sstd_data[0].sstd_column_factorial != NULL) {
        free(sstd_data[0].sstd_column_factorial);
        sstd_data[0].sstd_column_factorial = NULL;
    }
    if(sstd_data[0].D_basis_matrix != NULL) {
        free(sstd_data[0].D_basis_matrix[0].values);
        free(sstd_data[0].D_basis_matrix[0].columns_index);
        free(sstd_data[0].D_basis_matrix[0].rows_ptr);
        free(sstd_data[0].D_basis_matrix);
        sstd_data[0].D_basis_matrix = NULL;

        free(sstd_data[0].R_coeff_skip[0].values);
        free(sstd_data[0].R_coeff_skip[0].columns_index);
        free(sstd_data[0].R_coeff_skip[0].rows_ptr);
        free(sstd_data[0].R_coeff_skip);
        sstd_data[0].R_coeff_skip = NULL;
    }
    if(sstd_data[0].all_dictionary_tableau != NULL) {
        free(sstd_data[0].all_dictionary_tableau[0].entries);
        free(sstd_data[0].all_dictionary_tableau);
        sstd_data[0].all_dictionary_tableau = NULL;
    }
    if(sstd_data[0].dictionary_hash != NULL) {
        if(sstd_data[0].dictionary_hash[0].copy_mode == 1) {
            free(sstd_data[0].dictionary_hash[0].pt_arr[0].entries);
        }
        free(sstd_data[0].dictionary_hash[0].pt_arr);
        free(sstd_data[0].dictionary_hash[0].data_arr);
        free(sstd_data[0].dictionary_hash);
        sstd_data[0].dictionary_hash = NULL;
    }
    if(sstd_data[0].straighten_cache_index_one != NULL) {
        free(sstd_data[0].straighten_cache_index_one);
        sstd_data[0].straighten_cache_index_one = NULL;
        free(sstd_data[0].straighten_cache_index_two);
        sstd_data[0].straighten_cache_index_two = NULL;
        for(int thread = 0; thread < _STRAIGHTEN_NUM_THREADS; thread++) {
            free(sstd_data[0].straighten_cache[thread].values);
            free(sstd_data[0].straighten_cache[thread].index);
            free(sstd_data[0].straighten_cache[thread].rows_ptr);
        }
        free(sstd_data[0].straighten_cache);
        sstd_data[0].straighten_cache = NULL;
    }
    sstd_data[0].max_shared_12 = 0;
    sstd_data[0].lower_bound_sstd = 0;
    sstd_data[0].num_dictionary_tableau = 0;
    sstd_data[0].num_sstd_tableau = 0;
    sstd_data[0].max_filling_value = 0;
}

struct tableau * random_sstd(struct sstd_data_c * sstd_data) {
    uint64_t r_tab = 0;
    if(RAND_MAX < sstd_data[0].num_sstd_tableau) {
        uint64_t a = rand();
        uint64_t b = rand();
        r_tab = ((a*(((uint64_t)RAND_MAX) + 1)) + b) % sstd_data[0].num_sstd_tableau;
    }
    else {
        r_tab = rand() % sstd_data[0].num_sstd_tableau;
    }
    return sstd_data[0].all_sstd_tableau + r_tab;
}

struct packed_tableau * random_dictionary_tableau(struct sstd_data_c * sstd_data) {
    uint64_t r_tab = 0;
    if(RAND_MAX < sstd_data[0].num_dictionary_tableau) {
        uint64_t a = rand();
        uint64_t b = rand();
        r_tab = ((a*(((uint64_t)RAND_MAX) + 1)) + b) % sstd_data[0].num_dictionary_tableau;
    }
    else {
        r_tab = rand() % sstd_data[0].num_dictionary_tableau;
    }
    return sstd_data[0].all_dictionary_tableau + r_tab;
}

void tableau_coeff_simplify(struct tableau * tableau_arr, uint32_t num_tableau, struct shape_data_c * s_data) {
    #ifdef _STRAIGHTEN_USE_QSORT_S
    qsort_s(tableau_arr, num_tableau, sizeof(struct tableau), sstd_tableau_colword_cmp_s, s_data);
    #else
    qsort_r(tableau_arr, num_tableau, sizeof(struct tableau), sstd_tableau_colword_cmp_r, s_data);
    #endif

    for(int tab = 0; tab < (num_tableau-1); tab++) {
        if(sstd_tableau_colword_cmp_s(s_data, tableau_arr + tab, tableau_arr + tab + 1) == 0) {
            tableau_arr[tab + 1].coefficient += tableau_arr[tab].coefficient;
            tableau_arr[tab].coefficient = 0;
        }
    }
}

int64_t __c_compute_rearrangement_coeff(const uint8_t *restrict all_monomials, uint8_t *restrict t_data_c, const struct sstd_data_c *restrict sstd_data, const struct shape_data_c *restrict s_data, const int col, const int sign) {
    #define CAN_ADD(insert_row, from_row) t_data_c[(insert_row * sstd_data[0].max_filling_value) + (all_monomials[s_data[0].col_row_to_box[(col * s_data[0].conjugate[0]) + from_row]]-1)] != 0
    #define ADD(insert_row, from_row)     t_data_c[(insert_row * sstd_data[0].max_filling_value) + (all_monomials[s_data[0].col_row_to_box[(col * s_data[0].conjugate[0]) + from_row]]-1)]--;    
    #define SUB(insert_row, from_row)     t_data_c[(insert_row * sstd_data[0].max_filling_value) + (all_monomials[s_data[0].col_row_to_box[(col * s_data[0].conjugate[0]) + from_row]]-1)]++;    
    
    int64_t coeff = 0;

    switch(s_data[0].conjugate[col])
        {
            case 2:
                if(CAN_ADD(0,0) && CAN_ADD(1,1))
                    {
                    if(straighten_unlikely(col==0))
                        { 
                        return (sign%2)==0 ? 1 : -1;
                        }
                    ADD(0,0);
                    ADD(1,1);
                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                    SUB(0,0);
                    SUB(1,1); 
                    }
                if(CAN_ADD(0,1) && CAN_ADD(1,0))
                    {
                    if(straighten_unlikely(col==0))
                        { 
                        return (sign%2)==0 ? -1 : 1;
                        } 
                    ADD(0,1);
                    ADD(1,0);
                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign + 1);
                    SUB(0,1);
                    SUB(1,0);
                    }
            break;
            case 3:
                if (CAN_ADD(0, 0)) {
                    if (CAN_ADD(1, 1)) {
                        if (CAN_ADD(2, 2)) {
                            if(straighten_unlikely(col==0))
                                { 
                                return (sign%2)==0 ? 1 : -1;
                                }
                            ADD(0,0);
                            ADD(1,1);
                            ADD(2,2);
                            coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                            SUB(0,0);
                            SUB(1,1);
                            SUB(2,2);
                        }
                    }
                    if (CAN_ADD(1, 2)) {
                        if (CAN_ADD(2, 1)) {
                            if(straighten_unlikely(col==0))
                                { 
                                return (sign%2)==0 ? -1 : 1;
                                }
                            ADD(0,0);
                            ADD(1,2);
                            ADD(2,1);
                            coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign + 1);
                            SUB(0,0);
                            SUB(1,2);
                            SUB(2,1);
                        }
                    }
                }
                if (CAN_ADD(0, 1)) {
                    if (CAN_ADD(1, 0)) {
                        if (CAN_ADD(2, 2)) {
                            if(straighten_unlikely(col==0))
                                { 
                                return (sign%2)==0 ? -1 : 1;
                                }
                            ADD(0,1);
                            ADD(1,0);
                            ADD(2,2);
                            coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                            SUB(0,1);
                            SUB(1,0);
                            SUB(2,2);
                        }
                    }
                    if (CAN_ADD(1, 2)) {
                        if (CAN_ADD(2, 0)) {
                            if(straighten_unlikely(col==0))
                                { 
                                return (sign%2)==0 ? 1 : -1;
                                }
                            ADD(0,1);
                            ADD(1,2);
                            ADD(2,0);
                            coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                            SUB(0,1);
                            SUB(1,2);
                            SUB(2,0);
                        }
                    }
                }
                if (CAN_ADD(0, 2)) {
                    if (CAN_ADD(1, 0)) {
                        if (CAN_ADD(2, 1)) {
                            if(straighten_unlikely(col==0))
                                { 
                                return (sign%2)==0 ? 1 : -1;
                                }
                            ADD(0,2);
                            ADD(1,0);
                            ADD(2,1);
                            coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                            SUB(0,2);
                            SUB(1,0);
                            SUB(2,1);
                        }
                    }
                    if (CAN_ADD(1, 1)) {
                        if (CAN_ADD(2, 0)) {
                            if(straighten_unlikely(col==0))
                                { 
                                return (sign%2)==0 ? -1 : 1;
                                }
                            ADD(0,2);
                            ADD(1,1);
                            ADD(2,0);
                            coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                            SUB(0,2);
                            SUB(1,1);
                            SUB(2,0);
                        }
                    }
                }
            break;
            case 4:
                if (CAN_ADD(0, 0)) {
                    if (CAN_ADD(1, 1)) {
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 3)) {
                                if(straighten_unlikely(col==0)) { 
                                    return (sign%2)==0 ? 1 : -1;
                                }
                                ADD(0,0);
                                ADD(1,1);
                                ADD(2,2);
                                ADD(3,3);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                SUB(0,0);
                                SUB(1,1);
                                SUB(2,2);
                                SUB(3,3);    
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 2)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? -1 : 1;
                                    }
                                ADD(0,0);
                                ADD(1,1);
                                ADD(2,3);
                                ADD(3,2);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                SUB(0,0);
                                SUB(1,1);
                                SUB(2,3);
                                SUB(3,2);
                            }
                        }
                    }
                    if (CAN_ADD(1, 2)) {
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 3)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? -1 : 1;
                                    }
                                ADD(0,0);
                                ADD(1,2);
                                ADD(2,1);
                                ADD(3,3);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                SUB(0,0);
                                SUB(1,2);
                                SUB(2,1);
                                SUB(3,3);
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 1)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? 1 : -1;
                                    }
                                ADD(0,0);
                                ADD(1,2);
                                ADD(2,3);
                                ADD(3,1);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                SUB(0,0);
                                SUB(1,2);
                                SUB(2,3);
                                SUB(3,1);
                            }
                        }
                    }
                    if (CAN_ADD(1, 3)) {
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 2)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? 1 : -1;
                                    }
                                ADD(0,0);
                                ADD(1,3);
                                ADD(2,1);
                                ADD(3,2);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                SUB(0,0);
                                SUB(1,3);
                                SUB(2,1);
                                SUB(3,2);
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 1)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? -1 : 1;
                                    }
                                ADD(0,0);
                                ADD(1,3);
                                ADD(2,2);
                                ADD(3,1);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                SUB(0,0);
                                SUB(1,3);
                                SUB(2,2);
                                SUB(3,1);
                            }
                        }
                    }
                }
                if (CAN_ADD(0, 1)) {
                    if (CAN_ADD(1, 0)) {
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 3)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? -1 : 1;
                                    }
                                ADD(0,1);
                                ADD(1,0);
                                ADD(2,2);
                                ADD(3,3);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                SUB(0,1);
                                SUB(1,0);
                                SUB(2,2);
                                SUB(3,3);
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 2)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? 1 : -1;
                                    }
                                ADD(0,1);
                                ADD(1,0);
                                ADD(2,3);
                                ADD(3,2);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                SUB(0,1);
                                SUB(1,0);
                                SUB(2,3);
                                SUB(3,2);
                            }
                        }
                    }
                    if (CAN_ADD(1, 2)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 3)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? 1 : -1;
                                    }
                                ADD(0,1);
                                ADD(1,2);
                                ADD(2,0);
                                ADD(3,3);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                SUB(0,1);
                                SUB(1,2);
                                SUB(2,0);
                                SUB(3,3);
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 0)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? -1 : 1;
                                    }
                                ADD(0,1);
                                ADD(1,2);
                                ADD(2,3);
                                ADD(3,0);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                SUB(0,1);
                                SUB(1,2);
                                SUB(2,3);
                                SUB(3,0);
                            }
                        }
                    }
                    if (CAN_ADD(1, 3)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 2)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? -1 : 1;
                                    }
                                ADD(0,1);
                                ADD(1,3);
                                ADD(2,0);
                                ADD(3,2);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                SUB(0,1);
                                SUB(1,3);
                                SUB(2,0);
                                SUB(3,2);
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 0)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? 1 : -1;
                                    }
                                ADD(0,1);
                                ADD(1,3);
                                ADD(2,2);
                                ADD(3,0);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                SUB(0,1);
                                SUB(1,3);
                                SUB(2,2);
                                SUB(3,0);
                            }
                        }
                    }
                }
                if (CAN_ADD(0, 2)) {
                    if (CAN_ADD(1, 0)) {
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 3)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? 1 : -1;
                                    }
                                ADD(0,2);
                                ADD(1,0);
                                ADD(2,1);
                                ADD(3,3);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                SUB(0,2);
                                SUB(1,0);
                                SUB(2,1);
                                SUB(3,3);
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 1)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? -1 : 1;
                                    }
                                ADD(0,2);
                                ADD(1,0);
                                ADD(2,3);
                                ADD(3,1);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                SUB(0,2);
                                SUB(1,0);
                                SUB(2,3);
                                SUB(3,1);
                            }
                        }
                    }
                    if (CAN_ADD(1, 1)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 3)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? -1 : 1;
                                    }
                                ADD(0,2);
                                ADD(1,1);
                                ADD(2,0);
                                ADD(3,3);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                SUB(0,2);
                                SUB(1,1);
                                SUB(2,0);
                                SUB(3,3);
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 0)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? 1 : -1;
                                    }
                                ADD(0,2);
                                ADD(1,1);
                                ADD(2,3);
                                ADD(3,0);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                SUB(0,2);
                                SUB(1,1);
                                SUB(2,3);
                                SUB(3,0);
                            }
                        }
                    }
                    if (CAN_ADD(1, 3)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 1)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? 1 : -1;
                                    }
                                ADD(0,2);
                                ADD(1,3);
                                ADD(2,0);
                                ADD(3,1);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                SUB(0,2);
                                SUB(1,3);
                                SUB(2,0);
                                SUB(3,1);
                            }
                        }
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 0)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? -1 : 1;
                                    }
                                ADD(0,2);
                                ADD(1,3);
                                ADD(2,1);
                                ADD(3,0);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                SUB(0,2);
                                SUB(1,3);
                                SUB(2,1);
                                SUB(3,0);
                            }
                        }
                    }
                }
                if (CAN_ADD(0, 3)) {
                    if (CAN_ADD(1, 0)) {
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 2)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? -1 : 1;
                                    }
                                ADD(0,3);
                                ADD(1,0);
                                ADD(2,1);
                                ADD(3,2);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                SUB(0,3);
                                SUB(1,0);
                                SUB(2,1);
                                SUB(3,2);
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 1)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? 1 : -1;
                                    }
                                ADD(0,3);
                                ADD(1,0);
                                ADD(2,2);
                                ADD(3,1);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                SUB(0,3);
                                SUB(1,0);
                                SUB(2,2);
                                SUB(3,1);
                            }
                        }
                    }
                    if (CAN_ADD(1, 1)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 2)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? 1 : -1;
                                    }
                                ADD(0,3);
                                ADD(1,1);
                                ADD(2,0);
                                ADD(3,2);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                SUB(0,3);
                                SUB(1,1);
                                SUB(2,0);
                                SUB(3,2);
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 0)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? -1 : 1;
                                    }
                                ADD(0,3);
                                ADD(1,1);
                                ADD(2,2);
                                ADD(3,0);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                SUB(0,3);
                                SUB(1,1);
                                SUB(2,2);
                                SUB(3,0);
                            }
                        }
                    }
                    if (CAN_ADD(1, 2)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 1)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? -1 : 1;
                                    }
                                ADD(0,3);
                                ADD(1,2);
                                ADD(2,0);
                                ADD(3,1);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                SUB(0,3);
                                SUB(1,2);
                                SUB(2,0);
                                SUB(3,1);
                            }
                        }
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 0)) {
                                if(straighten_unlikely(col==0))
                                    { 
                                    return (sign%2)==0 ? 1 : -1;
                                    }
                                ADD(0,3);
                                ADD(1,2);
                                ADD(2,1);
                                ADD(3,0);
                                coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                SUB(0,3);
                                SUB(1,2);
                                SUB(2,1);
                                SUB(3,0);
                            }
                        }
                    }
                }
            break;
            #if _STRAIGHTEN_COMPILE_TIME_LIMIT >= 5
            case 5:
                if (CAN_ADD(0, 0)) {
                    if (CAN_ADD(1, 1)) {
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 1);
                                    ADD(2, 2);
                                    ADD(3, 3);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 0);
                                    SUB(1, 1);
                                    SUB(2, 2);
                                    SUB(3, 3);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 1);
                                    ADD(2, 2);
                                    ADD(3, 4);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 0);
                                    SUB(1, 1);
                                    SUB(2, 2);
                                    SUB(3, 4);
                                    SUB(4, 3);
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 1);
                                    ADD(2, 3);
                                    ADD(3, 2);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 0);
                                    SUB(1, 1);
                                    SUB(2, 3);
                                    SUB(3, 2);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 1);
                                    ADD(2, 3);
                                    ADD(3, 4);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 0);
                                    SUB(1, 1);
                                    SUB(2, 3);
                                    SUB(3, 4);
                                    SUB(4, 2);
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 1);
                                    ADD(2, 4);
                                    ADD(3, 2);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 0);
                                    SUB(1, 1);
                                    SUB(2, 4);
                                    SUB(3, 2);
                                    SUB(4, 3);
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 1);
                                    ADD(2, 4);
                                    ADD(3, 3);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 0);
                                    SUB(1, 1);
                                    SUB(2, 4);
                                    SUB(3, 3);
                                    SUB(4, 2);
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 2)) {
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 2);
                                    ADD(2, 1);
                                    ADD(3, 3);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 0);
                                    SUB(1, 2);
                                    SUB(2, 1);
                                    SUB(3, 3);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 2);
                                    ADD(2, 1);
                                    ADD(3, 4);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 0);
                                    SUB(1, 2);
                                    SUB(2, 1);
                                    SUB(3, 4);
                                    SUB(4, 3);
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 2);
                                    ADD(2, 3);
                                    ADD(3, 1);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 0);
                                    SUB(1, 2);
                                    SUB(2, 3);
                                    SUB(3, 1);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 2);
                                    ADD(2, 3);
                                    ADD(3, 4);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 0);
                                    SUB(1, 2);
                                    SUB(2, 3);
                                    SUB(3, 4);
                                    SUB(4, 1);
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 2);
                                    ADD(2, 4);
                                    ADD(3, 1);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 0);
                                    SUB(1, 2);
                                    SUB(2, 4);
                                    SUB(3, 1);
                                    SUB(4, 3);
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 2);
                                    ADD(2, 4);
                                    ADD(3, 3);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 0);
                                    SUB(1, 2);
                                    SUB(2, 4);
                                    SUB(3, 3);
                                    SUB(4, 1);
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 3)) {
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 3);
                                    ADD(2, 1);
                                    ADD(3, 2);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 0);
                                    SUB(1, 3);
                                    SUB(2, 1);
                                    SUB(3, 2);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 3);
                                    ADD(2, 1);
                                    ADD(3, 4);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 0);
                                    SUB(1, 3);
                                    SUB(2, 1);
                                    SUB(3, 4);
                                    SUB(4, 2);
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 3);
                                    ADD(2, 2);
                                    ADD(3, 1);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 0);
                                    SUB(1, 3);
                                    SUB(2, 2);
                                    SUB(3, 1);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 3);
                                    ADD(2, 2);
                                    ADD(3, 4);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 0);
                                    SUB(1, 3);
                                    SUB(2, 2);
                                    SUB(3, 4);
                                    SUB(4, 1);
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 3);
                                    ADD(2, 4);
                                    ADD(3, 1);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 0);
                                    SUB(1, 3);
                                    SUB(2, 4);
                                    SUB(3, 1);
                                    SUB(4, 2);
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 3);
                                    ADD(2, 4);
                                    ADD(3, 2);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 0);
                                    SUB(1, 3);
                                    SUB(2, 4);
                                    SUB(3, 2);
                                    SUB(4, 1);
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 4)) {
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 4);
                                    ADD(2, 1);
                                    ADD(3, 2);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 0);
                                    SUB(1, 4);
                                    SUB(2, 1);
                                    SUB(3, 2);
                                    SUB(4, 3);
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 4);
                                    ADD(2, 1);
                                    ADD(3, 3);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 0);
                                    SUB(1, 4);
                                    SUB(2, 1);
                                    SUB(3, 3);
                                    SUB(4, 2);
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 4);
                                    ADD(2, 2);
                                    ADD(3, 1);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 0);
                                    SUB(1, 4);
                                    SUB(2, 2);
                                    SUB(3, 1);
                                    SUB(4, 3);
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 4);
                                    ADD(2, 2);
                                    ADD(3, 3);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 0);
                                    SUB(1, 4);
                                    SUB(2, 2);
                                    SUB(3, 3);
                                    SUB(4, 1);
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 4);
                                    ADD(2, 3);
                                    ADD(3, 1);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 0);
                                    SUB(1, 4);
                                    SUB(2, 3);
                                    SUB(3, 1);
                                    SUB(4, 2);
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 0);
                                    ADD(1, 4);
                                    ADD(2, 3);
                                    ADD(3, 2);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 0);
                                    SUB(1, 4);
                                    SUB(2, 3);
                                    SUB(3, 2);
                                    SUB(4, 1);
                                }
                            }
                        }
                    }
                }
                if (CAN_ADD(0, 1)) {
                    if (CAN_ADD(1, 0)) {
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 0);
                                    ADD(2, 2);
                                    ADD(3, 3);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 1);
                                    SUB(1, 0);
                                    SUB(2, 2);
                                    SUB(3, 3);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 0);
                                    ADD(2, 2);
                                    ADD(3, 4);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 1);
                                    SUB(1, 0);
                                    SUB(2, 2);
                                    SUB(3, 4);
                                    SUB(4, 3);
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 0);
                                    ADD(2, 3);
                                    ADD(3, 2);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 1);
                                    SUB(1, 0);
                                    SUB(2, 3);
                                    SUB(3, 2);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 0);
                                    ADD(2, 3);
                                    ADD(3, 4);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 1);
                                    SUB(1, 0);
                                    SUB(2, 3);
                                    SUB(3, 4);
                                    SUB(4, 2);
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 0);
                                    ADD(2, 4);
                                    ADD(3, 2);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 1);
                                    SUB(1, 0);
                                    SUB(2, 4);
                                    SUB(3, 2);
                                    SUB(4, 3);
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 0);
                                    ADD(2, 4);
                                    ADD(3, 3);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 1);
                                    SUB(1, 0);
                                    SUB(2, 4);
                                    SUB(3, 3);
                                    SUB(4, 2);
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 2)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 2);
                                    ADD(2, 0);
                                    ADD(3, 3);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 1);
                                    SUB(1, 2);
                                    SUB(2, 0);
                                    SUB(3, 3);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 2);
                                    ADD(2, 0);
                                    ADD(3, 4);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 1);
                                    SUB(1, 2);
                                    SUB(2, 0);
                                    SUB(3, 4);
                                    SUB(4, 3);
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 2);
                                    ADD(2, 3);
                                    ADD(3, 0);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 1);
                                    SUB(1, 2);
                                    SUB(2, 3);
                                    SUB(3, 0);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 2);
                                    ADD(2, 3);
                                    ADD(3, 4);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 1);
                                    SUB(1, 2);
                                    SUB(2, 3);
                                    SUB(3, 4);
                                    SUB(4, 0);
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 2);
                                    ADD(2, 4);
                                    ADD(3, 0);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 1);
                                    SUB(1, 2);
                                    SUB(2, 4);
                                    SUB(3, 0);
                                    SUB(4, 3);
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 2);
                                    ADD(2, 4);
                                    ADD(3, 3);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 1);
                                    SUB(1, 2);
                                    SUB(2, 4);
                                    SUB(3, 3);
                                    SUB(4, 0);
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 3)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 3);
                                    ADD(2, 0);
                                    ADD(3, 2);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 1);
                                    SUB(1, 3);
                                    SUB(2, 0);
                                    SUB(3, 2);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 3);
                                    ADD(2, 0);
                                    ADD(3, 4);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 1);
                                    SUB(1, 3);
                                    SUB(2, 0);
                                    SUB(3, 4);
                                    SUB(4, 2);
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 3);
                                    ADD(2, 2);
                                    ADD(3, 0);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 1);
                                    SUB(1, 3);
                                    SUB(2, 2);
                                    SUB(3, 0);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 3);
                                    ADD(2, 2);
                                    ADD(3, 4);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 1);
                                    SUB(1, 3);
                                    SUB(2, 2);
                                    SUB(3, 4);
                                    SUB(4, 0);
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 3);
                                    ADD(2, 4);
                                    ADD(3, 0);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 1);
                                    SUB(1, 3);
                                    SUB(2, 4);
                                    SUB(3, 0);
                                    SUB(4, 2);
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 3);
                                    ADD(2, 4);
                                    ADD(3, 2);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 1);
                                    SUB(1, 3);
                                    SUB(2, 4);
                                    SUB(3, 2);
                                    SUB(4, 0);
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 4)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 4);
                                    ADD(2, 0);
                                    ADD(3, 2);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 1);
                                    SUB(1, 4);
                                    SUB(2, 0);
                                    SUB(3, 2);
                                    SUB(4, 3);
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 4);
                                    ADD(2, 0);
                                    ADD(3, 3);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 1);
                                    SUB(1, 4);
                                    SUB(2, 0);
                                    SUB(3, 3);
                                    SUB(4, 2);
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 4);
                                    ADD(2, 2);
                                    ADD(3, 0);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 1);
                                    SUB(1, 4);
                                    SUB(2, 2);
                                    SUB(3, 0);
                                    SUB(4, 3);
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 4);
                                    ADD(2, 2);
                                    ADD(3, 3);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 1);
                                    SUB(1, 4);
                                    SUB(2, 2);
                                    SUB(3, 3);
                                    SUB(4, 0);
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 4);
                                    ADD(2, 3);
                                    ADD(3, 0);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 1);
                                    SUB(1, 4);
                                    SUB(2, 3);
                                    SUB(3, 0);
                                    SUB(4, 2);
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 1);
                                    ADD(1, 4);
                                    ADD(2, 3);
                                    ADD(3, 2);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 1);
                                    SUB(1, 4);
                                    SUB(2, 3);
                                    SUB(3, 2);
                                    SUB(4, 0);
                                }
                            }
                        }
                    }
                }
                if (CAN_ADD(0, 2)) {
                    if (CAN_ADD(1, 0)) {
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 0);
                                    ADD(2, 1);
                                    ADD(3, 3);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 2);
                                    SUB(1, 0);
                                    SUB(2, 1);
                                    SUB(3, 3);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 0);
                                    ADD(2, 1);
                                    ADD(3, 4);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 2);
                                    SUB(1, 0);
                                    SUB(2, 1);
                                    SUB(3, 4);
                                    SUB(4, 3);
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 0);
                                    ADD(2, 3);
                                    ADD(3, 1);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 2);
                                    SUB(1, 0);
                                    SUB(2, 3);
                                    SUB(3, 1);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 0);
                                    ADD(2, 3);
                                    ADD(3, 4);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 2);
                                    SUB(1, 0);
                                    SUB(2, 3);
                                    SUB(3, 4);
                                    SUB(4, 1);
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 0);
                                    ADD(2, 4);
                                    ADD(3, 1);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 2);
                                    SUB(1, 0);
                                    SUB(2, 4);
                                    SUB(3, 1);
                                    SUB(4, 3);
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 0);
                                    ADD(2, 4);
                                    ADD(3, 3);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 2);
                                    SUB(1, 0);
                                    SUB(2, 4);
                                    SUB(3, 3);
                                    SUB(4, 1);
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 1)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 1);
                                    ADD(2, 0);
                                    ADD(3, 3);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 2);
                                    SUB(1, 1);
                                    SUB(2, 0);
                                    SUB(3, 3);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 1);
                                    ADD(2, 0);
                                    ADD(3, 4);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 2);
                                    SUB(1, 1);
                                    SUB(2, 0);
                                    SUB(3, 4);
                                    SUB(4, 3);
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 1);
                                    ADD(2, 3);
                                    ADD(3, 0);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 2);
                                    SUB(1, 1);
                                    SUB(2, 3);
                                    SUB(3, 0);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 1);
                                    ADD(2, 3);
                                    ADD(3, 4);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 2);
                                    SUB(1, 1);
                                    SUB(2, 3);
                                    SUB(3, 4);
                                    SUB(4, 0);
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 1);
                                    ADD(2, 4);
                                    ADD(3, 0);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 2);
                                    SUB(1, 1);
                                    SUB(2, 4);
                                    SUB(3, 0);
                                    SUB(4, 3);
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 1);
                                    ADD(2, 4);
                                    ADD(3, 3);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 2);
                                    SUB(1, 1);
                                    SUB(2, 4);
                                    SUB(3, 3);
                                    SUB(4, 0);
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 3)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 3);
                                    ADD(2, 0);
                                    ADD(3, 1);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 2);
                                    SUB(1, 3);
                                    SUB(2, 0);
                                    SUB(3, 1);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 3);
                                    ADD(2, 0);
                                    ADD(3, 4);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 2);
                                    SUB(1, 3);
                                    SUB(2, 0);
                                    SUB(3, 4);
                                    SUB(4, 1);
                                }
                            }
                        }
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 3);
                                    ADD(2, 1);
                                    ADD(3, 0);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 2);
                                    SUB(1, 3);
                                    SUB(2, 1);
                                    SUB(3, 0);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 3);
                                    ADD(2, 1);
                                    ADD(3, 4);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 2);
                                    SUB(1, 3);
                                    SUB(2, 1);
                                    SUB(3, 4);
                                    SUB(4, 0);
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 3);
                                    ADD(2, 4);
                                    ADD(3, 0);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 2);
                                    SUB(1, 3);
                                    SUB(2, 4);
                                    SUB(3, 0);
                                    SUB(4, 1);
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 3);
                                    ADD(2, 4);
                                    ADD(3, 1);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 2);
                                    SUB(1, 3);
                                    SUB(2, 4);
                                    SUB(3, 1);
                                    SUB(4, 0);
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 4)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 4);
                                    ADD(2, 0);
                                    ADD(3, 1);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 2);
                                    SUB(1, 4);
                                    SUB(2, 0);
                                    SUB(3, 1);
                                    SUB(4, 3);
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 4);
                                    ADD(2, 0);
                                    ADD(3, 3);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 2);
                                    SUB(1, 4);
                                    SUB(2, 0);
                                    SUB(3, 3);
                                    SUB(4, 1);
                                }
                            }
                        }
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 4);
                                    ADD(2, 1);
                                    ADD(3, 0);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 2);
                                    SUB(1, 4);
                                    SUB(2, 1);
                                    SUB(3, 0);
                                    SUB(4, 3);
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 4);
                                    ADD(2, 1);
                                    ADD(3, 3);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 2);
                                    SUB(1, 4);
                                    SUB(2, 1);
                                    SUB(3, 3);
                                    SUB(4, 0);
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 4);
                                    ADD(2, 3);
                                    ADD(3, 0);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 2);
                                    SUB(1, 4);
                                    SUB(2, 3);
                                    SUB(3, 0);
                                    SUB(4, 1);
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 2);
                                    ADD(1, 4);
                                    ADD(2, 3);
                                    ADD(3, 1);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 2);
                                    SUB(1, 4);
                                    SUB(2, 3);
                                    SUB(3, 1);
                                    SUB(4, 0);
                                }
                            }
                        }
                    }
                }
                if (CAN_ADD(0, 3)) {
                    if (CAN_ADD(1, 0)) {
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 0);
                                    ADD(2, 1);
                                    ADD(3, 2);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 3);
                                    SUB(1, 0);
                                    SUB(2, 1);
                                    SUB(3, 2);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 0);
                                    ADD(2, 1);
                                    ADD(3, 4);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 3);
                                    SUB(1, 0);
                                    SUB(2, 1);
                                    SUB(3, 4);
                                    SUB(4, 2);
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 0);
                                    ADD(2, 2);
                                    ADD(3, 1);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 3);
                                    SUB(1, 0);
                                    SUB(2, 2);
                                    SUB(3, 1);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 0);
                                    ADD(2, 2);
                                    ADD(3, 4);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 3);
                                    SUB(1, 0);
                                    SUB(2, 2);
                                    SUB(3, 4);
                                    SUB(4, 1);
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 0);
                                    ADD(2, 4);
                                    ADD(3, 1);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 3);
                                    SUB(1, 0);
                                    SUB(2, 4);
                                    SUB(3, 1);
                                    SUB(4, 2);
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 0);
                                    ADD(2, 4);
                                    ADD(3, 2);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 3);
                                    SUB(1, 0);
                                    SUB(2, 4);
                                    SUB(3, 2);
                                    SUB(4, 1);
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 1)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 1);
                                    ADD(2, 0);
                                    ADD(3, 2);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 3);
                                    SUB(1, 1);
                                    SUB(2, 0);
                                    SUB(3, 2);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 1);
                                    ADD(2, 0);
                                    ADD(3, 4);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 3);
                                    SUB(1, 1);
                                    SUB(2, 0);
                                    SUB(3, 4);
                                    SUB(4, 2);
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 1);
                                    ADD(2, 2);
                                    ADD(3, 0);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 3);
                                    SUB(1, 1);
                                    SUB(2, 2);
                                    SUB(3, 0);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 1);
                                    ADD(2, 2);
                                    ADD(3, 4);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 3);
                                    SUB(1, 1);
                                    SUB(2, 2);
                                    SUB(3, 4);
                                    SUB(4, 0);
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 1);
                                    ADD(2, 4);
                                    ADD(3, 0);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 3);
                                    SUB(1, 1);
                                    SUB(2, 4);
                                    SUB(3, 0);
                                    SUB(4, 2);
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 1);
                                    ADD(2, 4);
                                    ADD(3, 2);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 3);
                                    SUB(1, 1);
                                    SUB(2, 4);
                                    SUB(3, 2);
                                    SUB(4, 0);
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 2)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 2);
                                    ADD(2, 0);
                                    ADD(3, 1);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 3);
                                    SUB(1, 2);
                                    SUB(2, 0);
                                    SUB(3, 1);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 2);
                                    ADD(2, 0);
                                    ADD(3, 4);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 3);
                                    SUB(1, 2);
                                    SUB(2, 0);
                                    SUB(3, 4);
                                    SUB(4, 1);
                                }
                            }
                        }
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 4)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 2);
                                    ADD(2, 1);
                                    ADD(3, 0);
                                    ADD(4, 4);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 3);
                                    SUB(1, 2);
                                    SUB(2, 1);
                                    SUB(3, 0);
                                    SUB(4, 4);
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 2);
                                    ADD(2, 1);
                                    ADD(3, 4);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 3);
                                    SUB(1, 2);
                                    SUB(2, 1);
                                    SUB(3, 4);
                                    SUB(4, 0);
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 2);
                                    ADD(2, 4);
                                    ADD(3, 0);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 3);
                                    SUB(1, 2);
                                    SUB(2, 4);
                                    SUB(3, 0);
                                    SUB(4, 1);
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 2);
                                    ADD(2, 4);
                                    ADD(3, 1);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 3);
                                    SUB(1, 2);
                                    SUB(2, 4);
                                    SUB(3, 1);
                                    SUB(4, 0);
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 4)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 4);
                                    ADD(2, 0);
                                    ADD(3, 1);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 3);
                                    SUB(1, 4);
                                    SUB(2, 0);
                                    SUB(3, 1);
                                    SUB(4, 2);
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 4);
                                    ADD(2, 0);
                                    ADD(3, 2);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 3);
                                    SUB(1, 4);
                                    SUB(2, 0);
                                    SUB(3, 2);
                                    SUB(4, 1);
                                }
                            }
                        }
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 4);
                                    ADD(2, 1);
                                    ADD(3, 0);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 3);
                                    SUB(1, 4);
                                    SUB(2, 1);
                                    SUB(3, 0);
                                    SUB(4, 2);
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 4);
                                    ADD(2, 1);
                                    ADD(3, 2);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 3);
                                    SUB(1, 4);
                                    SUB(2, 1);
                                    SUB(3, 2);
                                    SUB(4, 0);
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 4);
                                    ADD(2, 2);
                                    ADD(3, 0);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 3);
                                    SUB(1, 4);
                                    SUB(2, 2);
                                    SUB(3, 0);
                                    SUB(4, 1);
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 3);
                                    ADD(1, 4);
                                    ADD(2, 2);
                                    ADD(3, 1);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 3);
                                    SUB(1, 4);
                                    SUB(2, 2);
                                    SUB(3, 1);
                                    SUB(4, 0);
                                }
                            }
                        }
                    }
                }
                if (CAN_ADD(0, 4)) {
                    if (CAN_ADD(1, 0)) {
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 0);
                                    ADD(2, 1);
                                    ADD(3, 2);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 4);
                                    SUB(1, 0);
                                    SUB(2, 1);
                                    SUB(3, 2);
                                    SUB(4, 3);
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 0);
                                    ADD(2, 1);
                                    ADD(3, 3);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 4);
                                    SUB(1, 0);
                                    SUB(2, 1);
                                    SUB(3, 3);
                                    SUB(4, 2);
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 0);
                                    ADD(2, 2);
                                    ADD(3, 1);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 4);
                                    SUB(1, 0);
                                    SUB(2, 2);
                                    SUB(3, 1);
                                    SUB(4, 3);
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 0);
                                    ADD(2, 2);
                                    ADD(3, 3);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 4);
                                    SUB(1, 0);
                                    SUB(2, 2);
                                    SUB(3, 3);
                                    SUB(4, 1);
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 0);
                                    ADD(2, 3);
                                    ADD(3, 1);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 4);
                                    SUB(1, 0);
                                    SUB(2, 3);
                                    SUB(3, 1);
                                    SUB(4, 2);
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 0);
                                    ADD(2, 3);
                                    ADD(3, 2);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 4);
                                    SUB(1, 0);
                                    SUB(2, 3);
                                    SUB(3, 2);
                                    SUB(4, 1);
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 1)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 1);
                                    ADD(2, 0);
                                    ADD(3, 2);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 4);
                                    SUB(1, 1);
                                    SUB(2, 0);
                                    SUB(3, 2);
                                    SUB(4, 3);
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 1);
                                    ADD(2, 0);
                                    ADD(3, 3);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 4);
                                    SUB(1, 1);
                                    SUB(2, 0);
                                    SUB(3, 3);
                                    SUB(4, 2);
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 1);
                                    ADD(2, 2);
                                    ADD(3, 0);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 4);
                                    SUB(1, 1);
                                    SUB(2, 2);
                                    SUB(3, 0);
                                    SUB(4, 3);
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 1);
                                    ADD(2, 2);
                                    ADD(3, 3);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 4);
                                    SUB(1, 1);
                                    SUB(2, 2);
                                    SUB(3, 3);
                                    SUB(4, 0);
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 1);
                                    ADD(2, 3);
                                    ADD(3, 0);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 4);
                                    SUB(1, 1);
                                    SUB(2, 3);
                                    SUB(3, 0);
                                    SUB(4, 2);
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 1);
                                    ADD(2, 3);
                                    ADD(3, 2);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 4);
                                    SUB(1, 1);
                                    SUB(2, 3);
                                    SUB(3, 2);
                                    SUB(4, 0);
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 2)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 2);
                                    ADD(2, 0);
                                    ADD(3, 1);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 4);
                                    SUB(1, 2);
                                    SUB(2, 0);
                                    SUB(3, 1);
                                    SUB(4, 3);
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 2);
                                    ADD(2, 0);
                                    ADD(3, 3);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 4);
                                    SUB(1, 2);
                                    SUB(2, 0);
                                    SUB(3, 3);
                                    SUB(4, 1);
                                }
                            }
                        }
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 3)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 2);
                                    ADD(2, 1);
                                    ADD(3, 0);
                                    ADD(4, 3);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 4);
                                    SUB(1, 2);
                                    SUB(2, 1);
                                    SUB(3, 0);
                                    SUB(4, 3);
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 2);
                                    ADD(2, 1);
                                    ADD(3, 3);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 4);
                                    SUB(1, 2);
                                    SUB(2, 1);
                                    SUB(3, 3);
                                    SUB(4, 0);
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 2);
                                    ADD(2, 3);
                                    ADD(3, 0);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 4);
                                    SUB(1, 2);
                                    SUB(2, 3);
                                    SUB(3, 0);
                                    SUB(4, 1);
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 2);
                                    ADD(2, 3);
                                    ADD(3, 1);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 4);
                                    SUB(1, 2);
                                    SUB(2, 3);
                                    SUB(3, 1);
                                    SUB(4, 0);
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 3)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 3);
                                    ADD(2, 0);
                                    ADD(3, 1);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 4);
                                    SUB(1, 3);
                                    SUB(2, 0);
                                    SUB(3, 1);
                                    SUB(4, 2);
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 3);
                                    ADD(2, 0);
                                    ADD(3, 2);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 4);
                                    SUB(1, 3);
                                    SUB(2, 0);
                                    SUB(3, 2);
                                    SUB(4, 1);
                                }
                            }
                        }
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 3);
                                    ADD(2, 1);
                                    ADD(3, 0);
                                    ADD(4, 2);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 4);
                                    SUB(1, 3);
                                    SUB(2, 1);
                                    SUB(3, 0);
                                    SUB(4, 2);
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 3);
                                    ADD(2, 1);
                                    ADD(3, 2);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 4);
                                    SUB(1, 3);
                                    SUB(2, 1);
                                    SUB(3, 2);
                                    SUB(4, 0);
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? -1 : 1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 3);
                                    ADD(2, 2);
                                    ADD(3, 0);
                                    ADD(4, 1);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                    SUB(0, 4);
                                    SUB(1, 3);
                                    SUB(2, 2);
                                    SUB(3, 0);
                                    SUB(4, 1);
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if(straighten_unlikely(col==0)) {
                                        return (sign%2)==0 ? 1 : -1;
                                    }
                                    ADD(0, 4);
                                    ADD(1, 3);
                                    ADD(2, 2);
                                    ADD(3, 1);
                                    ADD(4, 0);
                                    coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                    SUB(0, 4);
                                    SUB(1, 3);
                                    SUB(2, 2);
                                    SUB(3, 1);
                                    SUB(4, 0);
                                }
                            }
                        }
                    }
                }
                break;
            #endif
            #if _STRAIGHTEN_COMPILE_TIME_LIMIT >= 6
            case 6:
                if (CAN_ADD(0, 0)) {
                    if (CAN_ADD(1, 1)) {
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 2)) {
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 3)) {
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 4)) {
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 5)) {
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 0);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 0);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                    }
                }
                if (CAN_ADD(0, 1)) {
                    if (CAN_ADD(1, 0)) {
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 2)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 3)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 4)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 5)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 1);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 1);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                    }
                }
                if (CAN_ADD(0, 2)) {
                    if (CAN_ADD(1, 0)) {
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 1)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 3)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 4)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 5)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 2);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 2);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                    }
                }
                if (CAN_ADD(0, 3)) {
                    if (CAN_ADD(1, 0)) {
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 1)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 2)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 4);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 4)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 4);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 4);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 5)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 3);
                                        ADD(1, 5);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 3);
                                        SUB(1, 5);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                    }
                }
                if (CAN_ADD(0, 4)) {
                    if (CAN_ADD(1, 0)) {
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 0);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 0);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 1)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 1);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 1);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 2)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 2);
                                        ADD(2, 5);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 2);
                                        SUB(2, 5);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 3)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 1);
                                        ADD(3, 5);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 1);
                                        SUB(3, 5);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 5);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 5);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 5)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 5);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 5);
                                    }
                                }
                                if (CAN_ADD(4, 5)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 5);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 5);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 5)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 2);
                                        ADD(3, 5);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 2);
                                        SUB(3, 5);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 5)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 3);
                                        ADD(2, 5);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 3);
                                        SUB(2, 5);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 5)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 4);
                                        ADD(1, 5);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 4);
                                        SUB(1, 5);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                    }
                }
                if (CAN_ADD(0, 5)) {
                    if (CAN_ADD(1, 0)) {
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 0);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 0);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 1)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 2)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 2);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 2);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 2);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 2);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 2);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 2);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 2);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 2);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 2);
                                    }
                                }
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 1);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 2);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 1);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 2);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 2)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 0);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 0);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 1)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 3);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 3);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 3);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 1);
                                        ADD(3, 4);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 1);
                                        SUB(3, 4);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 3)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 0);
                                        ADD(4, 4);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 0);
                                        SUB(4, 4);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 4)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 3);
                                        ADD(3, 4);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 3);
                                        SUB(3, 4);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                        if (CAN_ADD(2, 4)) {
                            if (CAN_ADD(3, 0)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 1);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 1);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 0);
                                        ADD(4, 3);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 0);
                                        SUB(4, 3);
                                        SUB(5, 1);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 3)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 0);
                                        ADD(5, 3);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 0);
                                        SUB(5, 3);
                                    }
                                }
                                if (CAN_ADD(4, 3)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 1);
                                        ADD(4, 3);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 1);
                                        SUB(4, 3);
                                        SUB(5, 0);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 3)) {
                                if (CAN_ADD(4, 0)) {
                                    if (CAN_ADD(5, 1)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 0);
                                        ADD(5, 1);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 0);
                                        SUB(5, 1);
                                    }
                                }
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 0)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 2);
                                        ADD(2, 4);
                                        ADD(3, 3);
                                        ADD(4, 1);
                                        ADD(5, 0);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 2);
                                        SUB(2, 4);
                                        SUB(3, 3);
                                        SUB(4, 1);
                                        SUB(5, 0);
                                    }
                                }
                            }
                        }
                    }
                    if (CAN_ADD(1, 3)) {
                        if (CAN_ADD(2, 0)) {
                            if (CAN_ADD(3, 1)) {
                                if (CAN_ADD(4, 2)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? 1 : -1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 2);
                                        ADD(5, 4);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign);
                                        SUB(0, 5);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 2);
                                        SUB(5, 4);
                                    }
                                }
                                if (CAN_ADD(4, 4)) {
                                    if (CAN_ADD(5, 2)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 1);
                                        ADD(4, 4);
                                        ADD(5, 2);
                                        coeff = coeff + __c_compute_rearrangement_coeff(all_monomials, t_data_c, sstd_data, s_data, col-1, sign+1);
                                        SUB(0, 5);
                                        SUB(1, 3);
                                        SUB(2, 0);
                                        SUB(3, 1);
                                        SUB(4, 4);
                                        SUB(5, 2);
                                    }
                                }
                            }
                            if (CAN_ADD(3, 2)) {
                                if (CAN_ADD(4, 1)) {
                                    if (CAN_ADD(5, 4)) {
                                        if(straighten_unlikely(col==0)) {
                                            return (sign%2)==0 ? -1 : 1;
                                        }
                                        ADD(0, 5);
                                        ADD(1, 3);
                                        ADD(2, 0);
                                        ADD(3, 2);
                                        ADD(4, 1);
                   