import Pkg;
deps = Pkg.dependencies()
installs = Vector{String}()
for (uuid, dep) in deps
  dep.is_direct_dep || continue
  push!(installs,dep.name)
end

if !in("DynamicPolynomials",installs) Pkg.add("DynamicPolynomials")end
if !in("Combinatorics",installs) Pkg.add("Combinatorics")end
if !in("AbstractAlgebra",installs) Pkg.add("AbstractAlgebra")end
if !in("Random",installs) Pkg.add("Random")end
if !in("MutableArithmetics",installs) Pkg.add("MutableArithmetics")end


using DynamicPolynomials
using Combinatorics
using AbstractAlgebra
using Random
using MutableArithmetics
#using Dates #for timestamps in debug messages

variableDict = Dict{Vector{Int},PolyVar{true}}()
rescaledVariableDict = Dict{Vector{Int},Polynomial{true,BigInt}}()

function removeTrailingZeros(l::Vector{Int})::Vector{Int}
  ret = Array{Int,1}(undef,0)
  autoappend = false
  for i in reverse(1:length(l))
    if autoappend || l[i]!=0
      ret = vcat([l[i]],ret)
      autoappend = true
    end
  end
  return ret
end

function appendZeros(l::Vector{Int}, totallength::Int)::Vector{Int}
  ret = l
  while length(l) < totallength
    l = vcat(l,[0])
  end
  return l
end

function updateVariableDicts(numberofvariables::Int,innerdegree::Int)
  for l in multiexponents(numberofvariables,innerdegree)
    s = string("v")
    varindex = 1
    for i in l
      for j in 1:i
        s = string(s,"x")
        s = string(s,varindex)
      end
      varindex = varindex + 1
    end
    l = removeTrailingZeros(l)
    if !(l in keys(variableDict))
      eval(Meta.parse(string("@polyvar ",s)))
      variableDict[l] = eval(Meta.parse(s))
      rescaledVariableDict[l] = (factorial(innerdegree)÷multinomial(l...))*eval(Meta.parse(string(s,"+BigInt(0)")))
    end
  end
end

function hwvfilenamefromdatabase(rowlengths::Vector{Int}, rowwisecontent::Vector{Int}, hwvdatabasefolder::String)
  innerdegree = length(filter(x->x==1 ? true : false,rowwisecontent))
  outerdegree = length(Set(rowwisecontent))
  filename = string(hwvdatabasefolder,"/PlethysmDB",outerdegree,"/Inner",innerdegree,"/",replace(string(rowlengths)," "=>""),replace(string(rowwisecontent)," "=>""),".hwv")
  return filename
end

@polyvar DUMMYPOLYVAR
function tableau2hwv_checkandappenddatabase(rowlengths::Vector{Int}, rowwisecontent::Vector{Int}, hwvdatabasefolder::String)
  return parse(BigInt,"0")*DUMMYPOLYVAR+parse(BigInt,"0")
  innerdegree = length(filter(x->x==1 ? true : false,rowwisecontent))
  outerdegree = length(Set(rowwisecontent))
  filename = string(hwvdatabasefolder,"/PlethysmDB",outerdegree,"/Inner",innerdegree,"/",replace(string(rowlengths)," "=>""),replace(string(rowwisecontent)," "=>""),".hwv")
  if isfile(filename)
    columnlengths = [length(filter(x->x>=columnindex,rowlengths)) for columnindex in 1:rowlengths[1]]  #the transpose partition of rowlengths
    numberofvariables = columnlengths[1]
    updateVariableDicts(numberofvariables,innerdegree)
    #ret = toBigIntPoly(eval(Meta.parse(string(readline(filename),"+0")))) #apparently this is too tough when the polynomial is very large
    largestring = readline(filename)
    smallstrings = split(largestring,"+")
    ret = BigInt(0)+BigInt(0)*x[1]
    for smallstring in smallstrings
      part = toBigIntPoly(eval(Meta.parse(string(smallstring,"+0"))))
      MutableArithmetics.add!(ret,part)
    end
    return ret
  else
    print(string("Highest weight vector database file is missing: ",filename))
    exit(1)
  end
end










function evaluatehwvfromfile_slow(hwvfile::String,point::Polynomial{true,BigInt},numberofvariables::Int,innerdegree::Int)::BigInt
  if isfile(hwvfile)
    for l in multiexponents(numberofvariables,innerdegree)
      s = string("c")
      varindex = 1
      for i in l
        for j in 1:i
          s = string(s,"x")
          s = string(s,varindex)
        end
        varindex = varindex + 1
      end
      l = appendZeros(l,length(point.x.Z[1]))
      s = string(s," = BigInt(",point.a[findall(l_->l_==l,point.x.Z)[1]],")")
      eval(Meta.parse(s))
    end
    
    largestring = readline(hwvfile)
    largestring = replace(largestring,"v"=>"c")
    smallstrings = split(largestring,"+")
    ret = BigInt(0)
    for smallstring in smallstrings
      part = eval(Meta.parse(string(smallstring)))
      ret += part
    end
    return ret
  else
    error("hwv file not available")
  end
end









const BigInt_1 = parse(BigInt,"1")
const BigInt_0 = parse(BigInt,"0")
const BigInt_minus1 = parse(BigInt,"-1")
const UInt8_1 = parse(UInt8,"1")
const rx_times = r"\*"
const rx_acute = r"\^"

function monmonvalue(monmonstring::AbstractString,point_dict::Dict{String,BigInt})::BigInt
  if monmonstring[1]=='v'
    ret = BigInt_1
  else
    firsttimes = match(rx_times, monmonstring).offset
    ret = parse(BigInt,monmonstring[1:firsttimes-1])
    monmonstring = monmonstring[firsttimes+1:length(monmonstring)]
  end

  generaloffset = 1 #first symbol without a *
  
  last = false
  while !last
    times = match(rx_times, monmonstring, generaloffset)
    if times==nothing
      timesoffset = length(monmonstring)+1
      last=true
    else
      timesoffset = times.offset
    end
    monstring = monmonstring[generaloffset:timesoffset-1]
    
    exponent = UInt8_1
    acute = match(rx_acute, monstring)
    if acute!=nothing
      exponentstring = monstring[acute.offset+1:length(monstring)]
      exponent = parse(UInt8,exponentstring)
      monstringwithoutacute = monstring[1:acute.offset-1]
    else
      monstringwithoutacute = monstring
    end
    
    if haskey(point_dict,monstringwithoutacute)
      value = point_dict[monstringwithoutacute]
      while exponent>=1
        ret = ret * value
        exponent = exponent-1
      end
    else
      ret=BigInt_0
    end
    
    generaloffset = timesoffset+1
  end
  return ret
end

function evaluatefromfile(filename::String,point_dict::Dict{String,BigInt})
  file = open(filename, "r")
  line = read(file,String);
  line = strip(line);
  close(file)
  rx_plus = r"\+"
  rx_minus = r"-"
  rx_times = r"\*"

  generaloffset = 1 #search starts from this offset

  first = true
  last = false

  ret = BigInt_0
  
  while !last
    if first
      leftoffset = 1
      if line[1]=='-'
        rescalewithminus1=true
      else
        line = string("+",line)
        rescalewithminus1=false
      end
      first=false
    else
      plus = match(rx_plus, line, generaloffset)
      minus = match(rx_minus, line, generaloffset)
      if plus==nothing && minus==nothing
        println("ERROR")
      elseif plus==nothing && minus!=nothing
        leftoffset = minus.offset
        rescalewithminus1=true
      elseif plus!=nothing && minus==nothing
        leftoffset = plus.offset
        rescalewithminus1=false
      else
        plusoffset = plus.offset
        minusoffset = minus.offset
        leftoffset = min(plusoffset,minusoffset)
        if leftoffset==minusoffset
          rescalewithminus1=true
        else
          rescalewithminus1=false
        end
      end
    end

    plus = match(rx_plus, line, leftoffset+1)
    minus = match(rx_minus, line, leftoffset+1)
    if plus==nothing && minus==nothing
      rightoffset = length(line)+1
      last=true
    elseif plus==nothing && minus!=nothing
      rightoffset = minus.offset
    elseif plus!=nothing && minus==nothing
      rightoffset = plus.offset
    else
      plusoffset = plus.offset
      minusoffset = minus.offset
      rightoffset = min(plusoffset,minusoffset)
    end

  monmonstring = strip(line[leftoffset+1:rightoffset-1])
  product = monmonvalue(monmonstring,point_dict)
  if (rescalewithminus1)
    product = BigInt_minus1 * product
  end
  ret = ret + product
  
  generaloffset = rightoffset
  end
  return ret
end

function createdict(point::Polynomial{true,BigInt},numberofvariables::Int,innerdegree::Int)
  ret = Dict{String,BigInt}()
  for l in multiexponents(numberofvariables,innerdegree)
    s = string("v")
    varindex = 1
    for i in l
      for j in 1:i
        s = string(s,"x")
        s = string(s,varindex)
      end
      varindex = varindex + 1
    end
    l = appendZeros(l,length(point.x.Z[1]))
    fa = findall(l_->l_==l,point.x.Z)
    if length(fa)>0
      ret[s] = point.a[fa[1]]
    end
  end
  return ret
end

function evaluatehwvfromfile_fast(hwvfile::String,point::Polynomial{true,BigInt},numberofvariables::Int,innerdegree::Int)::BigInt
  if isfile(hwvfile)
    point_dict = createdict(point,numberofvariables,innerdegree)
    return evaluatefromfile(hwvfile,point_dict)
  else
    error(string("hwv file not available: ",hwvfile))
  end
end





###############################################################################
### <linearalgebra>
###############################################################################

function sumBR(q1::Tuple{BigInt,BigInt},q2::Tuple{BigInt,BigInt})::Tuple{BigInt,BigInt}
  if q1[2]==BigInt(0)
    error("sumBR: q1 has zero denominator!")
  end
  if q1[2]<BigInt(0)
    error("sumBR: q1 has negative denominator!")
  end
  if q2[2]==BigInt(0)
    error("sumBR: q2 has zero denominator!")
  end
  if q2[2]<BigInt(0)
    error("sumBR: q2 has negative denominator!")
  end
  numer = q1[1]*q2[2]+q1[2]*q2[1]
  denom = q1[2]*q2[2]
  g = gcd(numer,denom)
  if denom//g < BigInt(0)
    g = g * BigInt(-1)
  end
  return (numer//g,denom//g)
end

function prodBR(q1::Tuple{BigInt,BigInt},q2::Tuple{BigInt,BigInt})::Tuple{BigInt,BigInt}
  if q1[2]==BigInt(0)
    error("sumBR: q1 has zero denominator!")
  end
  if q1[2]<BigInt(0)
    error("sumBR: q1 has negative denominator!")
  end
  if q2[2]==BigInt(0)
    error("sumBR: q2 has zero denominator!")
  end
  if q2[2]<BigInt(0)
    error("sumBR: q2 has negative denominator!")
  end
  numer = q1[1]*q2[1]
  denom = q1[2]*q2[2]
  g = gcd(numer,denom)
  if denom//g < BigInt(0)
    g = g * BigInt(-1)
  end
  return (numer//g,denom//g)
end

function multinverseBR(q::Tuple{BigInt,BigInt})::Tuple{BigInt,BigInt}
  if q[1]==BigInt(0)
    error("multinverseBR: Division by zero!")
  end
  if q[1] > BigInt(0)
    return (q[2],q[1])
  end
  if q[1] < BigInt(0)
    return (-q[2],-q[1])
  end
end

function matrixproductBR(M1::Array{Tuple{BigInt,BigInt},2},M2::Array{Tuple{BigInt,BigInt},2})::Array{Tuple{BigInt,BigInt},2}
  nrows1 = size(M1)[1]
  ncols1 = size(M1)[2]
  nrows2 = size(M2)[1]
  ncols2 = size(M2)[2]
  if ncols1 != nrows2
    error(string("error in matrixproduct: nrows1=",nrows1,",  ncols1=",ncols1,"!=",nrows2,"=nrows2,  ncols2=",ncols2))
  end
  ret = Array{Tuple{BigInt,BigInt},2}(undef,nrows1,ncols2)
  for row in 1:nrows1
    for col in 1:ncols2
      sumofproducts = (BigInt(0),BigInt(1))
      for k in 1:ncols1
        sumofproducts = sumBR(sumofproducts,prodBR(M1[row,k],M2[k,col]))
      end
      ret[row,col] = sumofproducts
    end
  end
  return ret
end

function transposeBR(M::Array{Tuple{BigInt,BigInt},2})::Array{Tuple{BigInt,BigInt},2}
  origrows = size(M)[1]
  origcols = size(M)[2]
  R = Array{Tuple{BigInt,BigInt},2}(undef,origcols,origrows)
  for r in 1:origrows
    for c in 1:origcols
      R[c,r] = M[r,c]
    end
  end
  return R
end

function elementarymatrixBR(n::Int64, row::Int64, col::Int64, entry::Tuple{BigInt,BigInt})::Array{Tuple{BigInt,BigInt},2}
  ret = Array{Tuple{BigInt,BigInt},2}(undef,n,n)
  for r in 1:n
    for c in 1:n
      if row==r && col==c
        ret[r,c]=entry
      elseif r==c
        ret[r,c]=(BigInt(1),BigInt(1))
      else
        ret[r,c]=(BigInt(0),BigInt(1))
      end
    end
  end
  return ret
end

function rowechelonBRandRank(M::Array{Tuple{BigInt,BigInt},2})::Tuple{Array{Tuple{BigInt,BigInt},2},Array{Tuple{BigInt,BigInt},2},Array{Tuple{BigInt,BigInt},2},Int64}
  nrows = size(M)[1]
  ncols = size(M)[2]
  R = Array{Tuple{BigInt,BigInt},2}(undef,nrows,ncols)
  for r in 1:nrows
    for c in 1:ncols
      R[r,c] = M[r,c]
    end
  end
  T = Array{Tuple{BigInt,BigInt},2}(undef,nrows,nrows)
  Tinverse = Array{Tuple{BigInt,BigInt},2}(undef,nrows,nrows)
  for r in 1:nrows
    for c in 1:nrows
      if r==c
        T[r,c]=(BigInt(1),BigInt(1))
        Tinverse[r,c]=(BigInt(1),BigInt(1))
      else
        T[r,c]=(BigInt(0),BigInt(1))
        Tinverse[r,c]=(BigInt(0),BigInt(1))
      end
    end
  end
  currentrow = 1
  for c in 1:ncols
    if currentrow<nrows #???
      if R[currentrow,c]==(BigInt(0),BigInt(1))
        foundrow = 0
        for testrow in currentrow+1:nrows
          if R[testrow,c]!=(BigInt(0),BigInt(1))
            foundrow = testrow
            break
          end
        end
        if foundrow != 0
          P = elementarymatrixBR(nrows,1,1,(BigInt(1),BigInt(1)))
          P[foundrow,foundrow]=(BigInt(0),BigInt(1))
          P[currentrow,currentrow]=(BigInt(0),BigInt(1))
          P[foundrow,currentrow]=(BigInt(1),BigInt(1))
          P[currentrow,foundrow]=(BigInt(1),BigInt(1))
          R = matrixproductBR(P,R)
          T = matrixproductBR(T,P)
          Tinverse = matrixproductBR(P,Tinverse)
        else
          continue
        end
      end
      for r in currentrow+1:nrows
        multiplier = prodBR(prodBR(R[r,c],multinverseBR(R[currentrow,c])),(BigInt(-1),BigInt(1)))    #-R[r,c]/R[currentrow,c]
        transf = elementarymatrixBR(nrows,r,currentrow,multiplier)
        transfinverse = elementarymatrixBR(nrows,r,currentrow,prodBR(multiplier,(BigInt(-1),BigInt(1))))
        R = matrixproductBR(transf,R)
        T = matrixproductBR(T,transfinverse)
        Tinverse = matrixproductBR(transf,Tinverse)
      end
      currentrow=currentrow+1
    end
  end
  rank = nrows
  for row in reverse(1:nrows)
    found = 0
    for col in 1:ncols
      if R[row,col] != (BigInt(0),BigInt(1))
        found = 1
      end
    end
    if found==0
      rank = rank-1
    end
  end
  return (R,T,Tinverse,rank)
end


function unitvectorBR(n::Int64, i::Int64)::Array{Tuple{BigInt,BigInt},2}
  ret = Array{Tuple{BigInt,BigInt},2}(undef,n,1)
  for r in 1:n
    if r==i
      ret[r,1]=(BigInt(1),BigInt(1))
    else
      ret[r,1]=(BigInt(0),BigInt(1))
    end
  end
  return ret
end

function kernelBR(M::Array{Tuple{BigInt,BigInt},2})::Array{Tuple{BigInt,BigInt},2}#M is 10x3
  transp = transposeBR(M)#transp is 3x10
  (rowech,T,Tinverse,rank) = rowechelonBRandRank(transp)#T and Tinverse are 3x3
  transpTinverse = transposeBR(Tinverse)#transpTinverse is 3x3
  n = size(transp)[1]#n is 3
  E = Array{Tuple{BigInt,BigInt},2}(undef,n,n-rank)#E is 3x0
  for r in 1:n
    for c in 1:n-rank
      if r==c+rank
        E[r,c] = (BigInt(1),BigInt(1))
      else
        E[r,c] = (BigInt(0),BigInt(1))
      end
    end
  end
  ret = matrixproductBR(transpTinverse,E)
  for c in 1:size(ret)[2]
    lcmdenom = BigInt(1)
    for r in 1:size(ret)[1]
      lcmdenom = lcm(lcmdenom,ret[r,c][2])
    end
    for r in 1:size(ret)[1]
      ret[r,c] = prodBR(ret[r,c],(lcmdenom,BigInt(1)))
    end
    gcdnumer = BigInt(0)
    for r in 1:size(ret)[1]
      gcdnumer = gcd(gcdnumer,ret[r,c][1])
    end
    for r in 1:size(ret)[1]
      ret[r,c] = prodBR(ret[r,c],multinverseBR((gcdnumer,BigInt(1))))
    end
  end
  return ret
end


#This is the function that is called from outside:
function kernelBigInt(M::Array{BigInt,2})::Array{BigInt,2}
  N = Array{Tuple{BigInt,BigInt},2}(undef,size(M)[1],size(M)[2])
  for r in 1:size(N)[1]
    for c in 1:size(N)[2]
      N[r,c] = (M[r,c],BigInt(1))
    end
  end
  K = kernelBR(N)
  ret = Array{BigInt,2}(undef,size(K)[1],size(K)[2])
  for r in 1:size(ret)[1]
    for c in 1:size(ret)[2]
      ret[r,c] = K[r,c][1]
      if K[r,c][2]!=BigInt(1)
        error("kernelBigInt: K[r,c][2]!=BigInt(1)")
      end
    end
  end
  return ret
end

###############################################################################
### </linearalgebra>
###############################################################################

randoffset = convert(Int64,round(time()))

function sample(p::Polynomial{true,BigInt})#return type is a function
  return r -> subs(p,y=>rand(MersenneTwister(randoffset+r),big.(-30:30),length(y)))
end

function hwvmatrixkernel(listofhwvs::Vector{Polynomial{true,BigInt}},listofhwvfiles::Vector{String},xybigintpolynomial::Polynomial{true,BigInt}, numberofvariables::Int,rowlengths::Vector{Int},innerdegree::Int)::Vector{Polynomial{true,BigInt}}
  dim = length(listofhwvs)
  M = Array{BigInt,2}(undef,dim,dim)
  for row in 1:dim
    for col in 1:dim
      M[row,col] = evaluatehwvfromfile_fast(listofhwvfiles[col],sample(xybigintpolynomial)(row),numberofvariables,innerdegree)
    end
  end
  K = kernelBigInt(M)
  #print("kernel computed: ") #debug
  #print(Dates.format(Dates.now(), "HH:MM")) #debug
  #print("\n") #debug
  kerdim = size(K)[2]
  kermat = K
  ret = []
  for i in 1:kerdim
    eqn = BigInt(0)+BigInt(0)*x[1]
    for j in 1:length(listofhwvs)
      MutableArithmetics.add!(eqn,kermat[j,i]*listofhwvs[j])
    end
    ret = [ret...,eqn]
  end
  return ret
end


function toBigIntPoly(p)::Polynomial{true,BigInt}
  pp0 = p+0 #to make sure it is a polynomial and not just a monomial or a term
  q = BigInt(0)*x[1]+BigInt(0) #initializes q as the zero in Polynomial{true,BigInt}
  for term in MultivariatePolynomials.terms(pp0)
    MutableArithmetics.add!(q,parse(BigInt,string(term.α))*term.x)
  end
  return q
end

###################################################################
### Functions that are exported:
###################################################################

function computeequations_partition(partition::Vector{Int64}, xypolynomial::Polynomial{true,Int64}, dbfolder::String, eqnfile::String)
  return computeequations_partition(partition,toBigIntPoly(xypolynomial),dbfolder,eqnfile)
end

function computeequations_partition(partition::Vector{Int64}, xybigintpolynomial::Polynomial{true,BigInt}, dbfolder::String, eqnfile::String)
  randoffset = convert(Int64,round(time()))
  
  innerdegree = sum(sample(xybigintpolynomial)(1).x[1].z)
  numberofboxes = sum(partition)
  if numberofboxes % innerdegree != 0
    error("The partition has "+numberofboxes+" boxes, but that is not divisible by the inner degree "+innerdegree+".")
  end
  
  outerdegree = Int64(numberofboxes // innerdegree)

  if length(findall(k->k==0,partition))>0
    error("Partitions cannot contain any zeros.")
  end
  
  if reverse(partition)!=sort(partition)
    error("Partitions must be sorted.")
  end
  
  txtfilestring = string(dbfolder,"/PlethysmDB",outerdegree,"/Inner",innerdegree,"/",replace(string(partition)," "=>""),".txt")
  s = readline(txtfilestring)
  (la,plethysmcoefficient,tableaux)=eval(Meta.parse(s))
  listofhwvs = []
  listofhwvfiles = []
  for T in tableaux
    listofhwvs = [listofhwvs... , tableau2hwv_checkandappenddatabase(la,T,dbfolder)]
    listofhwvfiles = [listofhwvfiles... , hwvfilenamefromdatabase(la,T,dbfolder)]
  end
  if length(listofhwvfiles)>0
    
    ker = hwvmatrixkernel(listofhwvs,listofhwvfiles,xybigintpolynomial,length(partition),la,innerdegree)
    kerneldim = length(ker)
    
    if kerneldim>0
      io = open(eqnfile, "a")
      write(io,string(la,". ",kerneldim," eqns. plethysm coefficient = ",plethysmcoefficient,"\n"))
      for eqn in ker
        #normalize before writing:
        #g = gcd(coefficients(eqn)...)   #this does not work if coefficients(eqn) is too large
        g = BigInt(0)
        for c in coefficients(eqn)
          g = gcd(g,c)
        end
        #rescaledeqn = BigInt(0)
        #for t in DynamicPolynomials.terms(eqn)
        #  rescaledeqn += (t.α÷g) * t.x
        #end             #this is inefficient. Better:
        rescaledeqn = BigInt(0)+BigInt(0)*x[1]
        for t in DynamicPolynomials.terms(eqn)
          MutableArithmetics.add!(rescaledeqn,(t.α÷g) * t.x)
        end
        write(io,string(replace(string(rescaledeqn),"//1*"=>""),"\n"))
      end
      close(io)
    end
  end
end


function computeequations_degree(degree::Int,maxrows::Int,xypolynomial::Polynomial{true,Int64}, dbfolder::String, eqnfile::String)
  return computeequations_degree(degree,maxrows,toBigIntPoly(xypolynomial),dbfolder,eqnfile)
end

function computeequations_degree(degree::Int,maxrows::Int,xybigintpolynomial::Polynomial{true,BigInt}, dbfolder::String, eqnfile::String)
  innerdegree = sum(sample(xybigintpolynomial)(1).x[1].z)
  folderstring = string(dbfolder,"/PlethysmDB",degree,"/Inner",innerdegree)
  files = readdir(folderstring)
  for file in files
    if endswith(file,".txt")
      s = readline(string(folderstring,"/",file))
      (la,plethysmcoefficient,tableaux)=eval(Meta.parse(s))
      if length(la)>maxrows
        continue
      end
      if (plethysmcoefficient>0)
        print(string(la," with plethysm coefficient ",plethysmcoefficient,"\n"));
        computeequations_partition(la, xybigintpolynomial, dbfolder, eqnfile)
      end
    end
  end
end

function computeequations_degrees(degree_start::Int,degree_end::Int,maxrows::Int,xypolynomial::Polynomial{true,Int64}, dbfolder::String, eqnfile::String)
  return computeequations_degrees(degree_start,degree_end,maxrows,toBigIntPoly(xypolynomial),dbfolder,eqnfile)
end

function computeequations_degrees(degree_start::Int,degree_end::Int,maxrows::Int,xybigintpolynomial::Polynomial{true,BigInt}, dbfolder::String, eqnfile::String)
  for degree in degree_start:degree_end
    io = open(eqnfile, "a")
    write(io,string("DEGREE ",degree,"\n"))
    print(string("DEGREE ",degree,"\n"))
    close(io)
    computeequations_degree(degree,maxrows,xybigintpolynomial,dbfolder,eqnfile)
  end
end


