package lrc;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;

/**
 * This class contains the main applet's functionality and visualization, except for the FlowPanel.
 * @author ciken
 *
 */
public class FlowApplet extends JApplet {
	private static final long serialVersionUID = 4299284645091687473L;
	
	JLabel lambdaLabel = new JLabel("lambda");
	JLabel muLabel = new JLabel("mu");
	JLabel nuLabel = new JLabel("nu");
	FlowPanel flowPanel = new FlowPanel();
	Vector<Integer> lambda = null,mu = null,nu = null;
	Flow f = null;
	Vector<Turn> path = null;
	int iteration = 0;
	boolean inputPossible=true;
	JButton solutionButton=null;
	JButton nextButton=null;
	JButton stepButton=null;

	HashSet<Flow> expanded = null;
	HashSet<Flow> toExpand = null;
	Flow nextToExpand = null;
	boolean done = false;
	Long threshold = null;
	
	protected void foundFirstHiveFlow(Flow f) {
		flowPanel.label="LRC >= 1";
		expanded = new HashSet<Flow>();
		toExpand = new HashSet<Flow>();
		Flow ff = new Flow(f);
		ff.fix();
		toExpand.add(ff);
		nextToExpand = ff;
		done = false;
		threshold = null;
	}
	
	protected void foundNoHiveFlow() {
		expanded = null;
		toExpand = null;
		nextToExpand = null;
		done = false;
	}
	
	/**
	 * Partition labels and the flowPanel are adjusted and repainted.
	 */
	public void updateGraphics() {
		flowPanel.draw=false;
		flowPanel.f = f;
		flowPanel.path = path;
		flowPanel.lambda = lambda;
		flowPanel.mu = mu;
		flowPanel.nu = nu;
		flowPanel.iteration=iteration;
		String s = "lambda = ";
		String delim="";
		for (int i:lambda) {
			s+=delim+i;
			delim=",";
		}
		lambdaLabel.setText(s);
		s = "  mu = ";
		delim="";
		for (int i:mu) {
			s+=delim+i;
			delim=",";
		}
		muLabel.setText(s);
		s = "  nu = ";
		delim="";
		for (int i:nu) {
			s+=delim+i;
			delim=",";
		}
		nuLabel.setText(s);
		flowPanel.draw=true;
		if (solutionButton!=null) {
			if (iteration==-1) {
				solutionButton.setEnabled(false);
				stepButton.setEnabled(false);
				if (LRCPositivity.isCertificate(f, nu)) {
					if (done) {
						nextButton.setEnabled(false);
					} else {
						nextButton.setEnabled(true);
					}
				} else {
					nextButton.setEnabled(false);
				}
			}
			if (iteration>-1) {
				solutionButton.setEnabled(true);
				stepButton.setEnabled(true);
				nextButton.setEnabled(false);
			}
		}
		if (LRCPositivity.isCertificate(f, nu)) {
			if (expanded != null) {
				if (done) {
					if (expanded.size()>threshold && threshold!=0) {
						flowPanel.label="LRC > "+((expanded.size()-1)+", threshold reached.");
					} else {
						flowPanel.label="LRC = "+expanded.size();
					}
				} else {
					flowPanel.label="LRC >= "+(expanded.size()+1);
				}
			}
		}
		repaint();
	}
	
	private JButton addButton(String caption, JPanel buttonPanel) {
		buttonPanel.add(new JLabel(" "));//for some extra space
		JButton returnButton = new JButton(caption);buttonPanel.add(returnButton);returnButton.setAlignmentX(Component.CENTER_ALIGNMENT);
		return returnButton;
	}

	@Override
	public void init() {
		flowPanel = new FlowPanel();
		flowPanel.draw=false;
		lambda = new Vector<Integer>();lambda.add(4);lambda.add(3);lambda.add(2);lambda.add(2);lambda.add(2);lambda.add(1);lambda.add(1);
		mu = new Vector<Integer>();mu.add(5);mu.add(3);mu.add(2);mu.add(1);mu.add(1);mu.add(1);mu.add(0);
		nu = new Vector<Integer>();nu.add(7);nu.add(5);nu.add(4);nu.add(4);nu.add(4);nu.add(3);nu.add(1);
		f = new Flow(nu.size());
		path = null;
		iteration = 2;
		foundNoHiveFlow();
		updateGraphics();
		
		
		Container content = getContentPane();
		content.setBackground(Color.WHITE);
		content.setLayout(new BorderLayout()); 

		JPanel partitionLabelPanel = new JPanel();
		partitionLabelPanel.setBackground(Color.LIGHT_GRAY);
		partitionLabelPanel.add(lambdaLabel);
		partitionLabelPanel.add(muLabel);
		partitionLabelPanel.add(nuLabel);
		content.add(partitionLabelPanel,BorderLayout.NORTH);

		flowPanel.draw=false;
		content.add(flowPanel,BorderLayout.CENTER);
		flowPanel.f = f;
		flowPanel.lambda = lambda;
		flowPanel.mu = mu;
		flowPanel.nu = nu;
		flowPanel.path = path;

		JPanel buttonPanel = new JPanel();
		buttonPanel.setLayout(new BoxLayout(buttonPanel, BoxLayout.Y_AXIS));
		partitionLabelPanel.setBackground(Color.LIGHT_GRAY);
		JButton newpartitionsButton = addButton("new partitions", buttonPanel);
		JButton startagainButton = addButton("start again", buttonPanel);
		stepButton = addButton("step", buttonPanel);
		solutionButton = addButton("solution", buttonPanel);
		nextButton = addButton("next", buttonPanel);

		content.add(buttonPanel,BorderLayout.EAST);
		buttonPanel.setBackground(Color.LIGHT_GRAY);
		
		newpartitionsButton.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				if (inputPossible) {
					newpartitionsButtonClicked();
				}
			}
		});
		startagainButton.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				if (inputPossible) {
					flowPanel.draw=false;
					f = new Flow(nu.size());
					iteration = min_k(lambda,mu,nu);
					path = null;
					foundNoHiveFlow();
					updateGraphics();
				}
			}
		});
		solutionButton.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				if (inputPossible) {
					flowPanel.draw=false;
					f = LRCPositivity.decide(lambda, mu, nu);
					path = null;
					iteration=-1;
					if (LRCPositivity.isCertificate(f, nu)) foundFirstHiveFlow(f);
					updateGraphics();
				}
			}
		});
		stepButton.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				if (inputPossible) {
					if (path==null) {
						if (iteration>=0) {
							path = LRCPositivity.findShortestSTPath(f, iteration, lambda, mu, nu);
							if (path==null) iteration--;
							updateGraphics();
						}
					} else {
						long stepsize = (long)Math.pow(2, iteration);
						f.set(path.firstElement().getEndPos(), f.get(path.firstElement().getEndPos())+stepsize);
						for (Turn t : path) {
							f.set(t.getStartPos(), f.get(t.getStartPos())+stepsize);
						}
						if (LRCPositivity.findShortestSTPath(f, 0, lambda, mu, nu) == null) iteration=-1;
						path = null;
						updateGraphics();
					}
					if (LRCPositivity.isCertificate(f, nu)) foundFirstHiveFlow(f);
				}
			}
		});
		final FlowApplet parent = this;
		nextButton.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				if (inputPossible && toExpand!=null) {
					if (threshold == null) {
						String s = JOptionPane.showInputDialog(parent, "Please specify the threshold to which value you want to compute the LR-coefficient (0 for no threshold).");
						if (s==null) return;
						try {
							Long l = new Long(s);
							if (l<0) {
								JOptionPane.showMessageDialog(parent, "The threshold must be positive.", "Warning", JOptionPane.WARNING_MESSAGE);
								return;
							}
							threshold = l;
							updateGraphics();
						} catch (NumberFormatException nfe) {
							JOptionPane.showMessageDialog(parent, "The format was incorrect.", "Error", JOptionPane.ERROR_MESSAGE);
							return;
						}
						LRCCompute.flowCache=new HashSet<Flow>();
						Flow g = new Flow(f);
						g.fix();
						LRCCompute.flowCache.add(g);
					}
					toExpand.remove(nextToExpand);
					expanded.add(nextToExpand);
					if (threshold==0 || expanded.size()+toExpand.size()<=threshold) {
						Iterable <Flow> neighbors = null;
						if (threshold == 0) {
							neighbors = LRCCompute.computeNeighbors(nextToExpand);
						} else {
							neighbors = LRCCompute.computeNeighbors(nextToExpand, threshold);
						}
						for (Flow neighbor : neighbors) {
							if (!expanded.contains(neighbor)) toExpand.add(neighbor);
						}
					}
					if (threshold > 0 && toExpand.size()==1 && toExpand.size()+expanded.size()>threshold) {
						expanded.addAll(toExpand);
						toExpand.clear();
						done=true;
						updateGraphics();
					}

					if (toExpand.isEmpty()) {
						done=true;
						updateGraphics();
					} else {
						nextToExpand = toExpand.iterator().next();
						f = new Flow(nextToExpand);
						updateGraphics();
					}
				}
			}
		});
		updateGraphics();
	}
	
	private int min_k(Vector<Integer> lambda, Vector<Integer> mu, Vector<Integer> nu) {
		return Math.min(Math.max((int)(Math.log(lambda.firstElement())/Math.log(2)), (int)(Math.log(mu.firstElement())/Math.log(2))), (int)(Math.log(nu.firstElement())/Math.log(2)));
	}
	
	public void newpartitionsButtonClicked() {
		inputPossible = false;
		try {
			Vector<Integer> newlambda = new Vector<Integer>();
			Vector<Integer> newmu = new Vector<Integer>();
			Vector<Integer> newnu = new Vector<Integer>();
			String s = JOptionPane.showInputDialog(this, "Input lambda in comma separated format. E.g. 4,3,2,2,2,1,1");
			if (s==null) {
				inputPossible = true;
				return;
			}
			if (s.contains("-")) throw new NumberFormatException();
			String [] sa = s.split(",");
			newlambda = new Vector<Integer>();
			int lastentry=Integer.MAX_VALUE;
			for (String t : sa) {
				t=t.trim();
				int v = new Integer(t);
				if (v!=0) newlambda.add(v);
				if (lastentry < v) {
					JOptionPane.showMessageDialog(this, "lambda must be nonincreasing.","Warning",JOptionPane.WARNING_MESSAGE);
					inputPossible = true;
					return;
				}
				lastentry = v;
			}
			s = JOptionPane.showInputDialog(this, "Input mu in comma separated format. E.g. 5,3,2,1,1,1");
			if (s==null) {
				inputPossible = true;
				return;
			}
			if (s.contains("-")) throw new NumberFormatException();
			sa = s.split(",");
			newmu = new Vector<Integer>();
			lastentry=Integer.MAX_VALUE;
			for (String t : sa) {
				t=t.trim();
				int v = new Integer(t);
				if (v!=0) newmu.add(v);
				if (lastentry < v) {
					JOptionPane.showMessageDialog(this, "mu must be nonincreasing.","Warning",JOptionPane.WARNING_MESSAGE);
					inputPossible = true;
					return;
				}
				lastentry = v;
			}
			Vector<Integer> sumpartition = new Vector<Integer>();
			for (int i=0;i<Math.max(newlambda.size(), newmu.size());i++) sumpartition.add(0);
			for (int i=0;i<newlambda.size();i++) sumpartition.set(i, sumpartition.get(i)+newlambda.get(i));
			for (int i=0;i<newmu.size();i++) sumpartition.set(i, sumpartition.get(i)+newmu.get(i));
			String delim="";
			String sumstring = "";
			for (int j : sumpartition) {
				sumstring += delim+j;
				delim=",";
			}

			int weight = 0;
			for (int i : newlambda) weight+=i;
			for (int i : newmu) weight+=i;

			s = JOptionPane.showInputDialog(this, "Input nu in comma separated format. E.g. "+sumstring+". The numbers must add up to "+weight);
			if (s==null) {
				inputPossible = true;
				return;
			}
			if (s.contains("-")) throw new NumberFormatException();
			sa = s.split(",");
			newnu = new Vector<Integer>();
			lastentry=Integer.MAX_VALUE;
			for (String t : sa) {
				t=t.trim();
				int v = new Integer(t);
				if (v!=0) newnu.add(v);
				if (lastentry < v) {
					JOptionPane.showMessageDialog(this, "nu must be nonincreasing.","Warning",JOptionPane.WARNING_MESSAGE);
					inputPossible = true;
					return;
				}
				lastentry = v;
			}
			int wnu = 0;
			for (int i : newnu) wnu+=i;
			if (weight != wnu) {
				JOptionPane.showMessageDialog(this, "nu must be a partition of exactly "+weight+" boxes.","Warning",JOptionPane.WARNING_MESSAGE);
				inputPossible = true;
				return;
			}
			if (newnu.size()>=newlambda.size() && newnu.size()>=newmu.size()) {
				while (newnu.size()>newlambda.size()) newlambda.add(0);
				while (newnu.size()>newmu.size()) newmu.add(0);
				flowPanel.draw=false;
				lambda = newlambda;
				mu = newmu;
				nu = newnu;
				iteration = min_k(lambda,mu,nu);
				path = null;
				f = new Flow(newnu.size());
				foundNoHiveFlow();
				updateGraphics();
			} else {
				JOptionPane.showMessageDialog(this, "The partition nu must be the longest one.","Warning",JOptionPane.WARNING_MESSAGE);
			}
		} catch (NumberFormatException e) {
			JOptionPane.showMessageDialog(this, "The format was incorrect.", "Error", JOptionPane.ERROR_MESSAGE);
		}
		inputPossible = true;
	}
}
