package lrc;

import java.util.Arrays;

/**
 * Represents a flow.
 * The flow data is MUTABLE until fix() is called.
 * Fixed flows can be used as set elements.
 * Recall from the Java API Specification 6 SE:
 * "Note: Great care must be exercised if mutable objects are used as set elements. The behavior of a set is not specified if the value of an object is changed in a manner that affects equals comparisons while the object is an element in the set."
 * 
 * @author ciken
 */
public final class Flow {
	private class ArrayPositionAndSign {
		public final int index;
		public final long sign;
		public ArrayPositionAndSign(int index, long sign) {
			this.index=index; this.sign=sign;
		}
	}
	/**
	 * the two mutable data fields:
	 */
	private long[] data;
	private boolean isMutable;
	/**
	 * n is the number of border edges of a side of the triangular graph.
	 */
	public final int n;
	/**
	 * constructs a mutable zero flow.
	 * @param n the number of edges on each side of the triangular graph.
	 */
	public Flow(int n) {
		this.n=n;
		data = new long[(n*(n+1)*3)/2];
		isMutable = true;
	}
	/**
	 * constructs a mutable copy of f.
	 */
	public Flow(Flow f) {
		this(f.n);
		data = (long[])f.data.clone();
		isMutable = true;
	}

	/**
	 * after calling fix(), the object becomes immutable. The behaviour of its equals() method changes appropriately. It can be placed in sets.
	 */
	public void fix() {
		isMutable=false;
	}
	private ArrayPositionAndSign getArrayPositionAndSign(ThroughputPosition tpos) {
		if (!tpos.vertex.isVertexOfBigTriangle(n)) throw new OutOfTriangleException(tpos.vertex);
		if (!tpos.secondVertex().isVertexOfBigTriangle(n)) throw new OutOfTriangleException(tpos.secondVertex());
		/*
		 * handle all three rotations:
		 */
		int rowsabove=0,vertex1col=0,vertex2col=0,offset=0;
		if (tpos.direction==HexagonalDirection.UP3 || tpos.direction==HexagonalDirection.DOWN9) {
			rowsabove = tpos.vertex.row;
			vertex1col=tpos.vertex.col;
			vertex2col=tpos.secondVertex().col;
			offset=0;
		}
		if (tpos.direction==HexagonalDirection.UPLEFT1 || tpos.direction==HexagonalDirection.DOWNRIGHT7) {
			rowsabove = n-tpos.vertex.col;
			vertex1col=n-1-tpos.vertex.row;
			vertex2col=n-1-tpos.secondVertex().row;
			offset=(n*(n+1))/2+1;
		}
		if (tpos.direction==HexagonalDirection.UPRIGHT5 || tpos.direction==HexagonalDirection.DOWNLEFT11) {
			rowsabove = n+tpos.vertex.col-tpos.vertex.row;
			vertex1col=tpos.vertex.col;
			vertex2col=tpos.secondVertex().col;
			offset=n*(n+1);
		}
		return new ArrayPositionAndSign(offset+((rowsabove-1)*rowsabove)/2+Math.min(vertex1col,vertex2col),
				vertex1col<vertex2col?1:-1);
	}
	/**
	 * Returns how much flow flows through a an edge of the triangular graph in a given direction.
	 * @param tpos the triangular edge with direction
	 * @return the amount of flow that flows through tpos
	 */
	public long get(ThroughputPosition tpos) {
		ArrayPositionAndSign aps = getArrayPositionAndSign(tpos);
		return data[aps.index]*aps.sign;
	}
	/**
	 * Set much flow flows through a an edge of the triangular graph in a given direction.
	 * @param tpos the triangular edge with direction
	 * @param value the amount of flow that flows through tpos
	 */
	public void set(ThroughputPosition tpos, long value) {
		if (!isMutable) throw new UnsupportedOperationException("Flow.set is not supported for fixed flows.");
		ArrayPositionAndSign aps = getArrayPositionAndSign(tpos);
		data[aps.index]=value*aps.sign;
	}
	/**
	 * @param tp the diagonal of the rhombus whose slack is computed.
	 * @return the slack of a rhombus with diagonal tp.
	 */
	public long getSlack(ThroughputPosition tp) {
		return get(tp.goccw())-get(tp.gocw().gocw().gocw().gocw().gocw());
	}
	/**
	 * TikZ-output to compile in LaTeX
	 */
	@Override
	public String toString() {
		return FlowTikzVisualizer.flow2String(this);
	}

	@Override
	public int hashCode() {
		return Arrays.hashCode(data);
	}
	
	@Override
	public boolean equals(Object o) {
		if (o instanceof Flow) {
			Flow f = (Flow)o;
			if (isMutable || f.isMutable) return this==o;
			return Arrays.equals(data, f.data);
		}
		return false;
	}
}
