package lrc;
/**
 * immutable class.
 * Consists of a reference vertex in the triangular grid and a HexagonalDirection.
 * This defines an adjacent vertex in the triagular grid, accessible via secondVertex().
 * @author ciken
 */
public final class ThroughputPosition {
	/**
	 * The reference vertex in the triangular grid.
	 */
	public final RowAndColumn vertex;
	/**
	 * The direction vector which emanates from the reference vertex "vertex".
	 */
	public final HexagonalDirection direction;
	public ThroughputPosition(RowAndColumn vertex, HexagonalDirection direction) {
		this.vertex=vertex;this.direction=direction;
	}
	/**
	 * Go one counterclockwise turn from this throughput position.
	 */
	public ThroughputPosition goccw() {
		return new ThroughputPosition(vertex,direction.ccw());
	}
	/**
	 * Go one clockwise turn from this throughput position.
	 */
	public ThroughputPosition gocw() {
		return new ThroughputPosition(RowAndColumn.sum(vertex, direction.ccw().getRowAndColumn()),direction.cw());
	}
	/**
	 * Point in the other direction.
	 */
	public ThroughputPosition reverse() {
		return gocw().goccw().goccw().goccw().goccw().goccw().gocw();
	}
	/**
	 * The second vertex of this ThroughputPosition, besides the reference vertex "vertex".
	 */
	public RowAndColumn secondVertex() {
		return RowAndColumn.sum(vertex, direction.getRowAndColumn());
	}
	/**
	 * Checks if this ThroughputPosition lies on an edge of the triangular graph.
	 * @param n the number of border edges of a side of the triangular graph.
	 * @return whether the reference vertex and its second vertex lie in the triangular graph.
	 */
	public boolean liesInBigTriangle(int n) {
		return vertex.isVertexOfBigTriangle(n) && secondVertex().isVertexOfBigTriangle(n);
	}
	@Override
	public int hashCode() {
		return vertex.hashCode()+31*direction.hashCode();
	}
	@Override
	public boolean equals(Object o) {
		if (o instanceof ThroughputPosition) {
			ThroughputPosition tpos = (ThroughputPosition)o;
			return tpos.vertex.equals(vertex) && tpos.direction.equals(direction);
		}
		return false;
	}
	@Override
	public String toString() {
		return vertex.toString()+","+direction.toString();
	}
}
