package lrc;
import java.util.*;

/**
 * Contains static methods for deciding positivity of LR-coefficients.
 * @author ciken
 */
public final class LRCPositivity {
	private LRCPositivity() {};

	/**
	 * Checks, whether f proves that the Littlewood-Richardson coefficient is positive.
	 * @param f
	 * @param nu
	 * @return
	 */
	public static boolean isCertificate(Flow f, Vector<Integer> nu) {
		long delta = 0;
		for (int i=0;i<f.n;i++) {
			delta-=f.get(new ThroughputPosition(new RowAndColumn(i, 0), HexagonalDirection.DOWNRIGHT7));
		}
		long normnu = 0;
		for (int i : nu) normnu+=(long)i;
		return delta==normnu;
	}

	
	/**
	 * Decides positivity of a Littlewood-Richardson coefficient by returning an optimal hive flow.
	 * Input vector sizes must be equal!
	 * @param lambda
	 * @param mu
	 * @param nu with |nu|=|lambda|+|mu|
	 * @return an optimal hive flow
	 */
	public static Flow decide(Vector<Integer> lambda, Vector<Integer> mu, Vector<Integer> nu) {
		if (lambda.size()!=nu.size()) throw new Error("lambda.size()!=nu.size()");
		if (mu.size()!=nu.size()) throw new Error("mu.size()!=nu.size()");
		{
			int wlamu=0;for (int l:lambda) wlamu+=l;for (int l:mu) wlamu+=l;
			int wnu=0;for (int l:nu) wnu+=l;
			if (wnu!=wlamu) throw new Error("need |lambda|+|mu|=|nu|");
		}
		int n = nu.size();
		Flow f = new Flow(n);
		
		for (int iteration = (int)(Math.log(nu.firstElement())/Math.log(2))+1; iteration>=0; iteration-- ) {
			boolean foundpath = true;
			while (foundpath) {
				Vector<Turn> path = findShortestSTPath(f, iteration, lambda, mu, nu);
				if (path!=null) {
					//add 2^it times the path to f:
					long stepsize = (long)Math.pow(2, iteration);
					f.set(path.firstElement().getEndPos(), f.get(path.firstElement().getEndPos())+stepsize);
					for (Turn t : path) {
						f.set(t.getStartPos(), f.get(t.getStartPos())+stepsize);
					}
				} else {
					foundpath = false;
				}
			}
		}
		return f;
	}

	/**
	 * Finds a shortest s-t-turnpath over which 2^it flow units can be augmented.
	 * @param f a flow
	 * @param iteration the binary logarithm of the step size
	 * @param lambda
	 * @param mu
	 * @param nu with |nu|=|lambda|+|mu|
	 * @return a shortest s-t-turnpath over which 2^it flow units can be augmented. Returns null, if there is no such s-t-turnpath.
	 */
	public static Vector<Turn> findShortestSTPath(Flow f, int iteration, Vector<Integer> lambda, Vector<Integer> mu, Vector<Integer> nu) {
		long stepsize = (long)Math.pow(2, iteration);
		LinkedList<Turn> toExpand = new LinkedList<Turn>();
		Vector<Turn> exits = new Vector<Turn>();
		//fill "toExpand" and "exits" with vertices that are stepsize away from their throughput limit:
		for (int i=0;i<f.n;i++) {
			ThroughputPosition tpos = new ThroughputPosition(new RowAndColumn(i+1,i+1),HexagonalDirection.DOWNLEFT11);
			long throughput = f.get(tpos);
			if (throughput+stepsize<=lambda.get(i)) {
				try {if (f.getSlack(new Turn(tpos,Turn.LeftRight.RIGHT).getEndPos())!=0) {toExpand.add(new Turn(tpos,Turn.LeftRight.LEFT));}} catch (OutOfTriangleException e) {toExpand.add(new Turn(tpos,Turn.LeftRight.LEFT));}
				toExpand.add(new Turn(tpos,Turn.LeftRight.RIGHT));
			}
			tpos = new ThroughputPosition(new RowAndColumn(f.n,f.n-1-i),HexagonalDirection.UP3);
			throughput = f.get(tpos);
			if (throughput+stepsize<=mu.get(i)) {
				try {if (f.getSlack(new Turn(tpos,Turn.LeftRight.RIGHT).getEndPos())!=0) {toExpand.add(new Turn(tpos,Turn.LeftRight.LEFT));}} catch (OutOfTriangleException e) {toExpand.add(new Turn(tpos,Turn.LeftRight.LEFT));}
				toExpand.add(new Turn(tpos,Turn.LeftRight.RIGHT));
			}
			tpos = new ThroughputPosition(new RowAndColumn(i+1,0),HexagonalDirection.UPLEFT1);
			throughput = f.get(tpos);
			if (throughput+stepsize<=nu.get(i)) {
				exits.add(new Turn(Turn.LeftRight.LEFT,tpos));
				exits.add(new Turn(Turn.LeftRight.RIGHT,tpos));
			}
		}
		//Breadth-first-search:
		HashSet<Turn> expanded = new HashSet<Turn>();
		HashMap<Turn,Turn> predecessor = new HashMap<Turn,Turn>();
		Vector<Turn> path = null;
		while (!toExpand.isEmpty()) {
			Turn t = toExpand.remove();
			if (exits.contains(t)) {
				//found a path
				path = new Vector<Turn>();
				path.add(t);
				while (predecessor.containsKey(t)) {
					t = predecessor.get(t);
					path.add(t);
				}
				break;
			}
			expanded.add(t);
			for (Turn.LeftRight course : Turn.LeftRight.values()) {
				Turn t1 = t.nextTurn(course);
				//check if it is valid to continue turn t with turn t1
				if (t1.liesInBigTriangle(f.n)) {
					if (course==Turn.LeftRight.LEFT) {
						try {
							if (f.getSlack(t.nextTurn(Turn.LeftRight.RIGHT).getEndPos())==0) continue;
						} catch (OutOfTriangleException e) {}
					} else if (t.course==Turn.LeftRight.RIGHT && course==Turn.LeftRight.RIGHT) {
						try {
							if (f.getSlack(t.getEndPos())==0) continue;
						} catch (OutOfTriangleException e) {}
					}
					if (!expanded.contains(t1)) {
						predecessor.put(t1, t);
						toExpand.add(t1);
					}
				};
			}
		}
		return path;
	}

}
